/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include <bit/obstream.h>
#include <bit/ibstream.h>
#include <sstream>
#include <iostream>
#include <iomanip>
#include <stdio.h>
#include <byteswap.h>

using namespace bit;
using namespace std;

void print_buf(const char* buf, int size);
void print_results(const unsigned char* rbuf, size_t rsize, const unsigned char* ebuf, size_t esize);

void run_msb_octet_test(obstream& b, stringstream& s, const unsigned char* tbuf, size_t tsize, const unsigned char* ebuf, size_t esize);
void run_lsb_octet_test(obstream& b, stringstream& s, const unsigned char* tbuf, size_t tsize, size_t toctets, const unsigned char* ebuf, size_t esize);
void run_lsb_bit_test(obstream& b, stringstream& s, const unsigned char* tbuf, size_t tsize, size_t tbits, const unsigned char* ebuf, size_t esize);
/*
void run_msb_bit_test(ibstream& b, int n, uint32_t e);
template <typename T> void run_lsb_bit_var_test(ibstream& b, int n, T e);
*/

int passed=0, failed=0;

int main() {
    unsigned char tbuf[] = { 0x12, 0x34, 0x56, 0x78, 0x9a, 0xbc, 0xde, 0xf1, 0xbb };
    unsigned char ebuf[] = {
      0x12, 0x34, 0x56,                 // msb octet test 1, 3 octets from tbuf[0]
      0x9a, 0xbc,                       // msb octet test 2, 2 octets from tbuf[4]
      0x00, 0x00, 0x12, 0x34, 0x56,     // lsb octet test 1, 5 octets with buf 3 octets from tbuf[0]
      0x78, 0x9a, 0xbc,                 // lsb octet test 2, 3 octets with buf 5 octets from tbuf[1]
      0x03,                             // lsb bit test 1, 2 bits with buf 1 from tbuf[8]
      0x1b,                             // lsb bit test 1, 5 bits with buf 1 from tbuf[8]
      0xbb,                             // lsb variable test 1
    };

    stringstream s(ios_base::in|ios_base::out|ios_base::binary);
    obstream b(s);

    /**************************************************************/
    cout << "MSB Octet Test\n";
    run_msb_octet_test(b, s, tbuf+0, 3, ebuf, 3);
    run_msb_octet_test(b, s, tbuf+4, 2, ebuf, 5);
    cout << endl;

    /**************************************************************/
    cout << "LSB Octet Test\n";
    run_lsb_octet_test(b, s, tbuf+0, 3, 5, ebuf, 10);
    run_lsb_octet_test(b, s, tbuf+1, 5, 3, ebuf, 13);
    cout << endl;

    /**************************************************************/
    /*
    cout << "MSB Bit Test\n";
    run_msb_bit_test(b, 4,  0x10000000);
    run_msb_bit_test(b, 16, 0x23450000);
    run_msb_bit_test(b, 12, 0x67800000);
    cout << endl;
    */

    /**************************************************************/
    cout << "LSB Bit Test\n";
    run_lsb_bit_test(b, s, tbuf+8, 1, 2, ebuf, 14);
    run_lsb_bit_test(b, s, tbuf+8, 1, 5, ebuf, 15);
    cout << endl;

    /**************************************************************/
    cout << "LSB Variable Bit Test\n";
    uint16_t x = 0x123456f6; // using last 2 bits or 10
    uint32_t y = 0x123456a7; // using last 3 bits or 111
    uint64_t z = 0x123456a3; // using last 3 bits or 011
    // put together creates octet 10111011 or 1011 1011 or 0xbb

    b << bits(2) << x;
    b << bits(3) << y;
    b << bits(3) << z;
    print_results( (const unsigned char*)s.str().c_str(), s.str().length(), ebuf, 16);
    cout << endl;

    cout << endl << dec
    << "******************************************************************************" << endl
    << "Summary:  " << passed << " passed, " << failed << " failed" << endl;
    if (failed)
        cout << "WARNING: " << failed << " TESTS FAILED!!!" << endl << endl;
    cout << endl;

    return 0;
}

void print_buf(const char* buf, int size) {
    cout << "0x ";
    for (int i=0; i < size; i++)
        if (buf[i] & 0x80)
            printf("%01x%01x ", (unsigned int)buf[i]>>4&0x0f, (unsigned int)buf[i]&0x0f);
        else
            printf("%02x ", buf[i]);
}

void print_results(const unsigned char* rbuf, size_t rsize, const unsigned char* ebuf, size_t esize) {
    int result;
    cout << "  Expected Buffer: ";
    print_buf( (const char*) ebuf, esize);
    cout << endl;
    cout << "  Output Buffer:   ";
    print_buf((const char*) rbuf, rsize);
    cout << endl;
    if ( rsize != esize)
        result = -1;
    else
        result = memcmp(rbuf, ebuf, esize);

    cout << "  result: " << ( (result==0)?"passed":"FAILED") << endl;
    (result==0) ? passed++: failed++;
}

void run_msb_octet_test(obstream& b, stringstream& s, const unsigned char* tbuf, size_t tsize, const unsigned char* ebuf, size_t esize) {
    b.write_octets(tbuf, tsize);
    print_results( (const unsigned char*)s.str().c_str(), s.str().length(), ebuf, esize);
}

void run_lsb_octet_test(obstream& b, stringstream& s, const unsigned char* tbuf, size_t tsize, size_t toctets, const unsigned char* ebuf, size_t esize) {
    b.write_octets(tbuf, tsize, toctets);
    print_results( (const unsigned char*)s.str().c_str(), s.str().length(), ebuf, esize);
}

void run_lsb_bit_test(obstream& b, stringstream& s, const unsigned char* tbuf, size_t tsize, size_t tbits, const unsigned char* ebuf, size_t esize) {
  b.write_bits(tbuf, tsize, tbits);
  b.flush(false);
  print_results( (const unsigned char*)s.str().c_str(), s.str().length(), ebuf, esize);
}

/*


void run_lsb_octet_test(ibstream& b, int n, uint32_t e) {
  int rn;
  uint32_t d = 0x00;
  rn = b.read_octets( (unsigned char*) &d, 4, n);
  print_results("octets", rn, d, e);
}

void run_msb_bit_test(ibstream& b, int n, uint32_t e) {
  int rn;
  uint32_t d = 0x00;
  rn = b.read_bits( (unsigned char*) &d, n);
  print_results("bits", rn, d, e);
}

void run_lsb_bit_test(ibstream& b, int n, uint32_t e) {
  int rn;
  uint32_t d = 0x00;
  rn = b.read_bits( (unsigned char*) &d, 4, n);
  print_results("bits", rn, d, e);
}
*/
