/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "obstream.h"

#include <cstring>

using namespace bit;


obstream::obstream():
m_output(NULL) {}

obstream::obstream(std::ostream& output):
m_output(&output) {}


obstream::~obstream() {
    flush();
}

size_t obstream::write_octets(const unsigned char* buf, size_t octets) {
    size_t octets_written = 0;
    unsigned char temp;
    char* p;
    const unsigned char* q;

    if (m_output == NULL) return 0;

    // fast case is when there are no leftoverbits and we can just do octet writes
    if (m_numleftoverbits == 0) {
        m_output->write( (char*)buf, octets);
        return octets;
    } else {
        // we have m_numleftoverbits left from the previous write operation and need to shift these
        // bits onto the data and save the last m_numleftoverbits of data for the next write
        for (q=buf; q < buf+octets; q++) {
            temp = m_leftoverbits << 8 - m_numleftoverbits; // shift up the bits in m_leftoverbits for bitwise-or
            temp |= *q >>
                    m_numleftoverbits;  // bitwise-or in the additional bits from this octet in the buffer
            m_output->put( (char)temp); // write the created octet output
            m_leftoverbits = *q & m_masks[m_numleftoverbits]; // get the new leftover bits from the buffer for next write
            octets_written++;
        }
    }
    return octets_written;
}

size_t obstream::write_octets(const unsigned char* buf, size_t bufsize, size_t octets) {
    size_t owritten;
    const unsigned char* start;
    size_t to_write;

    if (m_output == NULL) return 0;

    if (octets > bufsize) {                                 // check to see if we need to zero pad
        for (size_t i = bufsize; i < octets; i++)           // output leading zeros since this is LSB ordering
            m_output->put('\0');
        start = buf;                                        // we'll start from the beginning of the buffer
        to_write = bufsize;
    } else {                                              // but, in this case we'll start somewhere in the middle
        start = buf + bufsize - octets;
        to_write = octets;
    }

    owritten = write_octets(start, to_write);                     // write into stream as MSB
    return owritten;
}

size_t obstream::write_bits(const unsigned char* buf, size_t n) {
    size_t whole_octets, woctets_written, bits_written;
    short unsigned int last_bits;
    const unsigned char* last_octet;

    if (m_output == NULL) return 0;

    // start by writing all whole octets
    // this results in a potentially new set of leftover bits
    // but does the right thing for the first n/8 octets
    whole_octets = n / 8;
    woctets_written = write_octets(buf, whole_octets);
    bits_written = woctets_written * 8;

    // now we need to write n%8 more bits from the last octet
    last_bits = n % 8;

    // we only need to continue if we have more bits to write
    if (last_bits) {
      last_octet = buf + whole_octets;
        // do we have to write another octet or just adjust m_leftoverbits?
      if (last_bits + m_numleftoverbits >= 8) {
        m_leftoverbits <<= 8 - m_numleftoverbits; // shift to upper bits
        m_leftoverbits |= *last_octet >> m_numleftoverbits; // bitwise-or in some of the other bits
        m_output->put(m_leftoverbits); // write out this bytes
        m_numleftoverbits = last_bits + m_numleftoverbits - 8; // calculate how many extra bits we have
        m_leftoverbits = *last_octet >> 8 - last_bits; // move down the unused lower bits
        m_leftoverbits &= m_masks[m_numleftoverbits]; // mask off the used upper bits leaving the unused upper bits
        }
        else {
          m_leftoverbits <<= last_bits; // shift up to make room for bits
          m_leftoverbits |= *last_octet >> 8 - last_bits; // bitwise-or in the bits once they're shifted to LSB position
          m_numleftoverbits += last_bits;
          }
          bits_written += last_bits;
    }

    return bits_written;
}

size_t obstream::write_bits(const unsigned char* buf, size_t bufsize, size_t bits) {
  size_t whole_octets, woctets_written, bits_written=0, remaining_bits = bits;
  short unsigned int first_bits;
  const unsigned char* first_octet;
  unsigned char c='\0';

  if (m_output == NULL) return 0;

  // do we need to start with some zero-padding?
  if (bits > bufsize*8) {
    bits_written = write_octets(&c, 1, bits/8 - bufsize) * 8; // write out as many whole octets as necessary
    bits_written += write_bits(&c, (bits-bufsize)%8); // write out any remaining bits in MSB ordering
    remaining_bits = bufsize * 8;
  }

  // we need to write bits%8 bits from the first octet
  first_octet = buf;
  first_bits = remaining_bits % 8;

  // do we even have any first bits
  if (first_bits) {
    first_octet += bufsize; // go to the end of the buffer
    first_octet -= bits/8 + (bits%8)?1:0; // and backtrack all whole octets plus a potential partial octet

    // do we have to write a complete octet or just adjust m_leftoverbits?
    if (first_bits + m_numleftoverbits >= 8) {
      m_leftoverbits <<= 8 - m_numleftoverbits; // shift to upper bits
      m_leftoverbits |= (*first_octet >> (first_bits - (8-m_numleftoverbits))) & m_masks[8-m_numleftoverbits]; // bitwise-or in some of the other bits
      m_output->put(m_leftoverbits); // write out this bytes
      m_numleftoverbits = first_bits + m_numleftoverbits - 8; // calculate how many extra bits we have
      m_leftoverbits = *first_octet & m_masks[m_numleftoverbits]; // mask off the used upper bits
    }
    else {
      m_leftoverbits <<= first_bits; // shift up to make room for bits
      m_leftoverbits |= *first_octet & m_masks[first_bits]; // bitwise-or in the bits once they're shifted to LSB position
      m_numleftoverbits += first_bits;
    }
    bits_written += first_bits;
    first_octet++;
  }

  // now we can write out the remainder of the octets
  bits_written += write_octets(first_octet, remaining_bits/8);

  return bits_written;
}

void obstream::flush(bool msb) {
  if (m_output == NULL) return;

  if (m_numleftoverbits == 0) // nothing to do
        return;
    if (msb) // we have left over bits and need to output in MSB
        m_output->put( (char) (m_leftoverbits << (8-m_numleftoverbits)) );
    else // we have left over bits and need to output in LSB
        m_output->put( (char)m_leftoverbits);

    // clear out left over bits values
    m_leftoverbits = 0x00;
    m_numleftoverbits = 0;
}


size_t obstream::write_octets(uint16_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_16(t);
    retval = write_octets( (unsigned char*) &t, 2, n);
    return retval;
}

size_t obstream::write_octets(uint32_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_32(t);
    retval = write_octets( (unsigned char*) &t, 4, n);
    return retval;
}

size_t obstream::write_octets(uint64_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_64(t);
    retval = write_octets( (unsigned char*) &t, 8, n);
    return retval;
}


size_t obstream::write_octets(int16_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_16(t);
    retval = write_octets( (unsigned char*) &t, 2, n);
    return retval;
}

size_t obstream::write_octets(int32_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_32(t);
    retval = write_octets( (unsigned char*) &t, 4, n);
    return retval;
}

size_t obstream::write_octets(int64_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_64(t);
    retval = write_octets( (unsigned char*) &t, 8, n);
    return retval;
}

size_t obstream::write_bits(uint16_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_16(t);
    retval = write_bits( (unsigned char*) &t, 2, n);
    return retval;
}

size_t obstream::write_bits(uint32_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_32(t);
    retval = write_bits( (unsigned char*) &t, 4, n);
    return retval;
}

size_t obstream::write_bits(uint64_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_64(t);
    retval = write_bits( (unsigned char*) &t, 8, n);
    return retval;
}


size_t obstream::write_bits(int16_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_16(t);
    retval = write_bits( (unsigned char*) &t, 2, n);
    return retval;
}

size_t obstream::write_bits(int32_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_32(t);
    retval = write_bits( (unsigned char*) &t, 4, n);
    return retval;
}

size_t obstream::write_bits(int64_t t, size_t n) {
    size_t retval;
    if (m_output == NULL) return 0;
    t = bswap_64(t);
    retval = write_bits( (unsigned char*) &t, 8, n);
    return retval;
}

void obstream::attach_stream(std::ostream& stream, bool clearbits) {
  m_output = &stream;
  if (clearbits) {
    m_numleftoverbits = 0;
    m_leftoverbits = 0x00;
  }
}

void obstream::detach_stream(bool clearbits) {
  m_output = NULL;
  if (clearbits) {
    m_numleftoverbits = 0;
    m_leftoverbits = 0x00;
  }
}
