/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef IBSTREAM_H
#define IBSTREAM_H

#include <istream>
#include <cstring>
#include <byteswap.h>

#include <bit/bstream.h>

namespace bit {


/**
@author Rick L Vinyard Jr
*/
/**
 * Manage an input stream with facilities for bitwise extraction
 **/
class ibstream: public bstream {
public:
    /**
    * Create an ibstream with no attached stream source.
     */
    ibstream();

    /**
     * Create an ibstream from a standard stream source.
     * The stream source will be used to retrieve actual data from.
     */
    ibstream(std::istream& input);

    /**
     * Performs any stream cleanup including output of any remaining bits.
     */
    ~ibstream();

    /**
     * MSB read of up to n octets into a memory buffer.
     * This method will only result in whole octet reads. Thus, if there
     * are 4 bits buffered, a request for 4 octets is made, and 2 octets are
     * actually available only 2 octets will be read with 4 bits still buffered.
     * @return the number of octets read
     */
    size_t read_octets(unsigned char* buf, size_t octets);

    /**
     * LSB read of up to n octets into a memory buffer of size m
     * with zero padding.
     * LSB, is not necessarily x86 little endian!
     * A read of 1234 into a 6 octet buffer results in 001234.
     * If n is greater than m, only m octets will be read.
     * @return the number of octets read.
     */
    size_t read_octets(unsigned char* buf, size_t bufsize, size_t octets);

    /**
     * MSB bit read of up to n bits into a memory buffer.
     * @return the number of bits read
     */
    size_t read_bits(unsigned char* buf, size_t bits);

    /**
     * LSB bit read of up to n bits into a memory buffer of size m octets
     * with zero padding.
     * No adjustment is made for x86 integer or any other architectures or types.
     * @return the number of bits read
     */
    size_t read_bits(unsigned char* buf, size_t bufsize, size_t bits);

    /**
     * LSB read of up to n octets into variable t with zero padding.
     * For architecture specific integer types the "right" thing is done to result
     * in a usable value.
     * The default is to read the entire variable off the stream.
     */
    template <typename T>
    size_t read_octets(T& t, size_t n=sizeof(T));

    size_t read_octets(uint16_t& t, size_t n=2);
    size_t read_octets(uint32_t& t, size_t n=4);
    size_t read_octets(uint64_t& t, size_t n=8);
    size_t read_octets(int16_t& t, size_t n=2);
    size_t read_octets(int32_t& t, size_t n=4);
    size_t read_octets(int64_t& t, size_t n=8);


    /**
     * LSB read of up to n bits into variable t with zero padding.
     * For architecture specific integer types the "right" thing is done to result
     * in a usable value.
     * The default is to read the entire variable off the stream.
     */
    template <typename T>
    size_t read_bits(T& t, size_t n=sizeof(T)*8);

    size_t read_bits(uint16_t& t, size_t n=2);
    size_t read_bits(uint32_t& t, size_t n=4);
    size_t read_bits(uint64_t& t, size_t n=8);
    size_t read_bits(int16_t& t, size_t n=2);
    size_t read_bits(int32_t& t, size_t n=4);
    size_t read_bits(int64_t& t, size_t n=8);


    friend ibstream& operator>>(ibstream& s, const bits& b);
    friend ibstream& operator>>(ibstream& s, const octets& o);
    friend ibstream& operator>>(ibstream& s, const whole& w);
    template <typename T>
    friend ibstream& operator>>(ibstream& s, T& t);

    /**
     * Set the associated input stream to stream.
     * @param clearbits If true any cached bits will be cleared, and if false (default) cached bits remain
     * available for reading.
     */
    void attach_stream(std::istream& stream, bool clearbits=false);

    /**
     * Remove the currently attached stream. Futher attempts to read will fail unless there are
     * cached bits.
     * @param clearbits If true any cached bits will be cleared, and if false (default) cached bits remain
     * available for reading.
     **/
    void detach_stream(bool clearbits=false);

protected:
    /** A pointer to the associated input stream. A reference is not used since
     * it is anticipated that future versions will provide a constructor that does not
     * require an associated stream.
     */
    std::istream* m_input;

};

template <typename T>
inline        size_t ibstream::read_octets(T& t, size_t n) {
  if (!m_input) return 0;

  if (n > sizeof(T))
        n = sizeof(T);
    return read_octets( (unsigned char*) &t, n, sizeof(T));
}

template <typename T>
inline size_t ibstream::read_bits(T& t, size_t n) {
  if (m_input == NULL && m_numleftoverbits == 0) return 0;

  if (n > sizeof(T) * 8 )
        n = sizeof(T) * 8;
   return read_bits( (unsigned char*) &t, n, sizeof(T));
}

inline ibstream& operator>>(ibstream& s, const bits& b) {
  s.m_state = bstream::BITS;
    s.m_stateval = b.val;
    return s;
}

inline ibstream& operator>>(ibstream& s, const octets& o) {
    s.m_state = bstream::OCTETS;
    s.m_stateval = o.val;
    return s;
}

inline ibstream& operator>>(ibstream& s, const whole& w) {
    s.m_state = bstream::WHOLE;
    s.m_stateval = 0;
    return s;
}


template <typename T>
inline ibstream& operator>>(ibstream& s, T& t) {
    switch (s.m_state) {
    case bstream::WHOLE:
        s.read_octets(t);
        break;
    case bstream::BITS:
        s.read_bits(t, s.m_stateval);
        break;
    case bstream::OCTETS:
        s.read_octets(t, s.m_stateval);
        break;
    }
    return s;
}


}; // namespace bit

#endif
