/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef BITFIELDBUFFER_H
#define BITFIELDBUFFER_H

#include <string>
#include <queue>
#include <set>

#include <bit/buffer.h>
#include <bit/tuple.h>
#include <bit/data.h>
#include <bit/error.h>

namespace bit
  {

  class FieldBuffer;

  class FieldProxy
    {
    public:

      FieldProxy(FieldBuffer& b, TupleBase& f): m_buffer(b), m_field(f)
      { }

      FieldProxy operator[](unsigned index)
      {
        return FieldProxy(m_buffer, m_field[index]);
      }

      FieldProxy operator[](std::string index)
      {
        return FieldProxy(m_buffer, m_field[index]);
      }
      
      template <typename T>
        operator T() {
          T t;
          unpack(t);
          return t;
        }
      
      /**
       * Returns a copy of the underlying data buffer allocated
       */
      Data data();

      bool unpack(void* mem, size_t mem_octets);

      template <typename T>
      bool unpack(T& val)
      {
        return unpack(&val, sizeof(T));
      }

      bool pack(const void* mem, size_t mem_octets);
      bool pack(const void* mem, size_t mem_octets, size_t n);

      template <typename T>
      bool pack(const T& val)
      {
        return pack(&val, sizeof(T));
      }

//       template <typename T>
//       bool pack(const T& val, size_t n)
//       {
//         return pack(&val, sizeof(T), n);
//       }

      template <typename T>
      FieldProxy& operator=(const T t)
      {
        pack(t);
        return *this;
      }


    protected:
      FieldBuffer& m_buffer;
      TupleBase& m_field;

    };

  /**
  @author Rick L. Vinyard, Jr.
  */
  class FieldBuffer : public Buffer
    {
    public:
      friend class ElementProxy;
      friend class FieldProxy;

      FieldBuffer(size_t initial_size=0, bool dynamic=true, size_t sizemax=0);
      FieldBuffer(Tuple& fields, size_t initial_size=0, bool dynamic=true, size_t sizemax=0);
      FieldBuffer(const uint8_t* external_data, size_t data_octets, bool dynamic=true, size_t sizemax=0);
      FieldBuffer(Tuple& fields, const uint8_t* external_data, size_t data_octets, bool dynamic=true, size_t sizemax=0);

      ~FieldBuffer();

      FieldProxy operator[](size_t);
      FieldProxy operator[](std::string);

      virtual void set_data(const void* data, size_t size);

      void set_fields(Tuple& fields) throw ();
      void clear_fields() throw ();
      Tuple& fields() throw (error::no_fields);

      FieldBuffer& operator=(const FieldBuffer& other);

      sigc::signal<void, std::string> signal_field_changed();

      sigc::signal<void> signal_fields_changed();

    protected:
      /**
       * A pointer to the fields associated with this buffer, or NULL
       * if there is no fields currently associated.
       */
      Tuple* m_fields;

      bool m_fields_set;

      /**
       * Causes the internal signal to be emitted notifying connected slots
       * that the data at field i has been changed.
       *
       * This method provides proper calling of the emit signal even when
       * a callback triggers a change. This would result in a recursive signal
       * which sigc++ doesn't allow.
       */
      void on_field_changed(std::string name);

      void on_fields_changed();

    private:

      sigc::signal<void> m_signal_fields_changed;

      /**
       * The signal emitted when an fieldsed value is changed.
       *
       * This member has been move to private, since derived children should not
       * directly emit the signal, but instead should call the notify_of_changes_to_fields
       * method.
       *
       * This method properly ensures that a signal is emitted for each changed fields,
       * even when a callback triggers an fields to change. This is necessary since
       * sigc++ does not allow recursive signals anymore.
       */
      sigc::signal<void, std::string> m_signal_field_changed;

      // Stores the fields that have changed and still need to have an emit signal
      //called. The fields are stored in the order they are changed.
      std::queue< std::string > m_signal_field_changed_queue;

      //Stores the set of fields that have changed and still need an emit signal.
      //The set is used to provide quick lookup O(lg n) rather than the O(n) necessary
      //in the queue.
      std::set< std::string >  m_signal_field_changed_set;

      bool  m_signal_field_changed_emitting;
    };

}

#endif
