/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "fieldbuffer.h"

using namespace bit;

FieldBuffer::FieldBuffer(size_t initial_size, bool dynamic, size_t sizemax):
    Buffer(initial_size, dynamic, sizemax),
    m_fields(NULL),
    m_fields_set(false),
    m_signal_field_changed_emitting(false)
{}

FieldBuffer::FieldBuffer(Tuple& fields, size_t initial_size, bool dynamic, size_t sizemax):
    Buffer(initial_size, dynamic, sizemax),
    m_fields(&fields),
    m_fields_set(true),
    m_signal_field_changed_emitting(false)
{}

FieldBuffer::FieldBuffer(const uint8_t* external_data, size_t data_octets, bool dynamic, size_t sizemax):
    Buffer(external_data, data_octets, dynamic, sizemax),
    m_fields(NULL),
    m_fields_set(false),
    m_signal_field_changed_emitting(false)
{}

FieldBuffer::FieldBuffer(Tuple& fields, const uint8_t* external_data, size_t data_octets, bool dynamic, size_t sizemax):
    Buffer(external_data, data_octets, dynamic, sizemax),
    m_fields(&fields),
    m_fields_set(true),
    m_signal_field_changed_emitting(false)
{}

FieldBuffer::~FieldBuffer()
{}

FieldProxy FieldBuffer::operator[](size_t field_id) {
  return FieldProxy(*this, fields()[field_id]);
}

FieldProxy FieldBuffer::operator[](std::string field_id) {
  return FieldProxy(*this, fields()[field_id]);
}

void FieldBuffer::set_fields(Tuple& fields ) throw ()
{
  m_fields = &fields;
  m_fields_set = true;
  on_fields_changed();
}

void FieldBuffer::clear_fields() throw ()
{
  m_fields = NULL;
  m_fields_set = false;
}

Tuple& FieldBuffer::fields( ) throw (error::no_fields)
{
  if (m_fields_set)
    return *m_fields;
  throw error::no_fields();
}

sigc::signal< void, std::string > FieldBuffer::signal_field_changed( )
{
  return m_signal_field_changed;
}

FieldBuffer& FieldBuffer::operator =( const FieldBuffer & other )
{
  m_fields = other.m_fields;
  Buffer::operator=(other);
  return *this;
}

void FieldBuffer::on_field_changed( std::string name )
{
  std::string current = name;

  if (m_signal_field_changed_emitting)
  {
      //we only need to mark each field for emitting once, so check to see if we already
      //have it queued for emitting
    if (m_signal_field_changed_set.find(current) == m_signal_field_changed_set.end())
    {
          //it's not already marked, so add it to both data structures
      m_signal_field_changed_set.insert(current);
      m_signal_field_changed_queue.push(current);
    }
    return;
  }

  m_signal_field_changed_emitting = true;
  do
  {
    if (m_signal_field_changed_set.size() > 0)
    {
      current = m_signal_field_changed_queue.front();
      m_signal_field_changed_queue.pop();
      m_signal_field_changed_set.erase(current);
    }
    m_signal_field_changed.emit(current);
  }
  while (m_signal_field_changed_set.size() > 0);

  m_signal_field_changed_emitting = false;
  return;
}

void FieldBuffer::on_fields_changed( )
{
  m_signal_fields_changed.emit();
}

sigc::signal<void> FieldBuffer::signal_fields_changed( )
{
  return m_signal_fields_changed;
}

void FieldBuffer::set_data( const void * data, size_t size )
{
  Buffer::set_data(data, size);
  m_signal_fields_changed.emit();
}

Data FieldProxy::data( )
{
  size_t size = m_field.bits();
  size = size / 8 + (size%8?1:0);
  Data d = Data( size );
  unpack(d.data.get(), size);
  return d;
}

bool FieldProxy::unpack( void * mem, size_t mem_octets )
{
  return m_buffer.unpack(mem, mem_octets, m_field.start(), m_field.bits());
}

bool FieldProxy::pack( const void * mem, size_t mem_octets )
{
  bool b = m_buffer.pack(mem, mem_octets, m_field.start(), m_field.bits(), m_field.bits());
  if (b)
    m_buffer.on_field_changed(m_field.full_name(m_field.depth()-1));
  return b;
}

bool FieldProxy::pack( const void * mem, size_t mem_octets, size_t n )
{
  bool b = m_buffer.pack(mem, mem_octets, m_field.start(), m_field.bits(), n);
  if (b)
    m_buffer.on_field_changed(m_field.full_name(m_field.depth()-1));
  return b;
}

