/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef BITBUFFER_H
#define BITBUFFER_H

#include <sigc++/sigc++.h>
#include <conexus/io.h>

#include <bit/data.h>

namespace bit
  {

  /**
   * todo: create an overloaded = operator that copies memory if the buffer is the owner
  @author Rick L. Vinyard, Jr.
  */
  class Buffer: public sigc::trackable
    {
    public:
      /**
       * Construct a buffer of a given size that is allowed to dynamically grown as
       * pack/unpack requests are made.
       */
      Buffer(size_t initial_size=0, bool dynamic=true, size_t sizemax=0);

      /**
       * Construct a buffer that is a copy of external data.
       */
      Buffer(const uint8_t* external_data, size_t data_octets, bool dynamic=true, size_t sizemax=0);

      ~Buffer();

      /**
       * Get the underlying data buffer.
       *
       * Note: if this is a dynamically resizeable buffer, you should connect to
       * the data_location_changed signal.
       */
      const uint8_t * const data() const;

      /**
       * Get the currently allocated size
       */
      size_t size() const;

      /**
       * Transfer data from the buffer area into the low order bits of a memory location.
       * @param mem The memory location to transfer buffer data into
       * @param mem_octets The size of the memory location in octets
       * @param buf_offset_bits The offset from the beginning of the buffer in bits to begin extraction from
       * @param bits The size of the data to extract in bits
       * @return true if transfer accomplished, false otherwise
       *
       * Bit Buffer
       * ==================================================
       * |000000000011111111112222222222333333333344444444|
       * |012345678901234567890123456789012345678901234567|
       * ==================================================
       *                ^ buf_offset_bits = 14
       *                ^ extract_bits = 20 ^
       *
       * Memory Buffer
       * ==================================================
       * |000000000011111111112222222222333333333344444444|
       * |012345678901234567890123456789012345678901234567|
       * ==================================================
       * ^ mem = start of memory address
       * ^ mem_octets = 6                                ^
       *                             ^ Resulting data    ^
       */
      bool unpack(void* mem, size_t mem_octets, size_t buf_offset_bits, size_t extract_bits);

      template <typename T>
      bool unpack(T& val, size_t buf_offset_bits, size_t extract_bits)
      {
        return unpack(&val, sizeof(T), buf_offset_bits, extract_bits);
      }

      /**
       * Transfer data from the low order bits of a memory location into the low order bits of a segment of the buffer area.
       * @param mem The memory location to transfer from
       * @param mem_octets The size of the memory location in octets
       * @param buf_offset The offset from the start of the data buffer in bits
       * @param buf_tgtsize The size of the target area in the buffer in bits
       * @param n The size of the data to extract in bits
       * @return true if transfer accomplished, false otherwise.
       * If @param offset + @param destsize are greater than the size of the buffer and the buffer
       * cannot be expanded to accommodate the request zero is returned.
       *
       * Memory Buffer
       * ==================================================
       * |000000000011111111112222222222333333333344444444|
       * |012345678901234567890123456789012345678901234567|
       * ==================================================
       * ^ mem = start of memory address
       * ^ mem_octets = 6                                ^
       *                                        ^ n = 10 ^
       *                                        ^ cpy out^
       *
       * Bit Buffer
       * ==================================================
       * |000000000011111111112222222222333333333344444444|
       * |012345678901234567890123456789012345678901234567|
       * ==================================================
       *                ^ buf_offset_bits = 14
       *                ^ buf_tgtsize  = 20 ^
       *                          ^ n = 10  ^
       *                          ^ cpy in  ^
       */
      bool pack(const void* mem, size_t mem_octets, size_t buf_offset, size_t buf_tgtsize, size_t n);

      template <typename T>
      bool pack(const T& val, size_t offset, size_t destsize, size_t n)
      {
        bool b;
        b = pack(&val, sizeof(T), offset, destsize, n);
        return b;
      }

      bool clear_bits(size_t offset, size_t bits)
      {
        return clear_bits(offset, bits, false);
      }

      void clear();

      /**
       * Returns true if the buffer is allowed to dynamically expand as necessary to accommodate pack requests.
       * If pack requests an offset and size that is beyond the bounds of the currently allocated buffer and
       * the buffer is set to dynamic, the buffer will be dynamically expanded and the expanded area cleared to zero
       * to accommodate the request.
       */
      bool is_dynamic() const;

      /**
       * If true, allow the buffer to dynamically expand as necessary to accommodate pack requests that are
       * beyond the bounds of the buffer.
       */
      void set_dynamic(bool b=true);

      /**
       * Sets the buffer to a copy of the memory location pointed at by data to be of octet size.
       */
      virtual void set_data(const void* data, size_t size);

      /**
       * Sets the underlying data buffer to the requested size. If size_request is 0, the buffer will
       * be freed. The new size is limited to the set maximum size, or unlimited if sizemax=0.
       *
       * @return The new size of the data buffer.
       */
      virtual size_t set_size(size_t size_request);

      /**
       * Gets the maximum size this buffer is allowed to grow
       */
      size_t get_sizemax();

      /**
       * Sets the maximum size that this buffer is allowed to grow.
       * Note: if sizemax is set to a value less than the current
       * buffer size the buffer will be truncated immediately.
       */
      void set_sizemax(size_t sizemax);

      /**
       * Overloaded = operator does 'the right thing' for the buffer chunk according to the ownership
       * semantics of the 'other' buffer.
       * If the other buffer does not own its memory chunk, then niether will this buffer, and if necessary
       * the old memory chunk will be cleaned up. This means that both memory buffers will operate on the
       * same piece of memory.
       *
       * If the other buffer did own its memory chunk then a copy will be made and the same semantics for
       * dynamic growth will apply.
       */
      Buffer& operator=(const Buffer& other);

      /**
       * Signals changes to the underlying data buffer location in memory. This most
       * likely occurs when the buffer is dynamically resized.
       */
      sigc::signal<void> signal_data_location_changed()
      {
        return m_signal_data_location_changed;
      }

      /**
       * Signals a change to the data within the buffer.
       */
      sigc::signal<void> signal_data_changed()
      {
        return m_signal_data_changed;
      }

      /**
       * Signals a change to the size of the internal buffer.
       */
      sigc::signal<void> signal_size_changed()
      {
        return m_signal_size_changed;
      }

    protected:
      uint8_t* m_pdata;
      size_t m_size;
      bool m_dynamic;
      size_t m_sizemax;

      /**
       * Utility method to clear the buffer
       *
       * on_changed* methods are not called since this is protected. Therefore it is the responsibility
       * of the calling method to call the appropriate on_changed* methods
       */
      void free_buffer();

      /**
       * Utility method to clear bits and suppress calling the on_changed* methods
       */
      bool clear_bits(size_t offset, size_t bits, bool suppress);

      virtual void on_data_location_changed();

      virtual void on_size_changed();

      virtual void on_data_changed();

    private:
      sigc::signal<void> m_signal_data_changed;
      sigc::signal<void> m_signal_data_location_changed;
      sigc::signal<void> m_signal_size_changed;

      bool m_signal_data_changed_need_reemit;
      bool m_signal_data_changed_emitting;


    };

}

conexus::IO& operator<<(conexus::IO&, bit::Buffer&);
conexus::IO& operator<<(conexus::IO&, bit::Data&);
conexus::IO& operator>>(conexus::IO&, bit::Buffer&);
conexus::IO& operator>>(conexus::IO&, bit::Data&);

#endif
