/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "buffer.h"

#include <assert.h>

#include <iostream>
#include <iomanip>
#include <algorithm>

#define OCTET_CEIL(x) ( (x)/8 + (( (x)%8 )?1:0) )
#define OCTET_FLOOR(x) ( (x)/8 )
#define UPPER_BITS(x) ( (x)%8 )
#define LOWER_BITS(x) ( (8 - UPPER_BITS(x))%8 )
#define OFFSET_PTR_FLOOR(x) ( m_pdata + OCTET_FLOOR(x) )
#define OFFSET_PTR_CEIL(x) ( m_pdata + OCTET_CEIL(x) )

using namespace bit;

const uint8_t masks[8] =
  {
    0x00, 0x01, 0x03, 0x07, 0x0F, 0x1F, 0x3F, 0x7F
  };

Buffer::Buffer(size_t size, bool dynamic, size_t sizemax):
    m_pdata(NULL),
    m_size(size),
    m_dynamic(dynamic),
    m_sizemax(sizemax),
    m_signal_data_changed_emitting(false),
    m_signal_data_changed_need_reemit(false)
{
  if (m_size > 0)
    m_pdata = (uint8_t*)calloc(1, m_size);
  // if we couldn't allocate the chunk, drop size to zero
  if (m_pdata == NULL)
    m_size = 0;
  m_sizemax = std::max(size, sizemax);
}

Buffer::Buffer( const uint8_t * external_data, size_t data_octets, bool dynamic, size_t sizemax ):
    m_size(data_octets),
    m_dynamic(dynamic),
    m_sizemax(sizemax)
{
  m_pdata = (uint8_t*) calloc( m_size, sizeof(uint8_t) );
  memcpy( m_pdata, external_data, m_size );
  m_sizemax = std::max(data_octets, sizemax);
}

Buffer::~Buffer()
{
  free_buffer();
}

const uint8_t * const Buffer::data() const
  {
    return m_pdata;
  }

size_t Buffer::size() const
  {
    return m_size;
  }

bool Buffer::is_dynamic() const
  {
    return m_dynamic;
  }

void Buffer::set_dynamic(bool b)
{
  m_dynamic = b;
}

bool Buffer::unpack(void* mem, size_t mem_octets, size_t offset, size_t extract)
{
  size_t ot, bsd, befpo, bsu, fwo, lwo, po;
  uint8_t mask, temp;
  uint8_t *ad, *p;
  size_t needed_size;

  // make basic assumptions: do we have a data buffer? are we extracting into a null address?
  if (   m_pdata == NULL || mem == NULL )
    return false;

  // are we being asked to extract more octets than the target can accept
  if ( mem_octets < OCTET_CEIL(extract) )
    // if so, reduce the request size
    extract = mem_octets * 8;

  // check to see if the request is beyond the current size
  if ( m_size < OCTET_CEIL(offset+extract) )
    {
      if (!m_dynamic )
        return false; // there's nothing we can do if we aren't dynamic

      // try to adjust the buffer size to accomodate the request
      needed_size = OCTET_CEIL(offset+extract);
      if (set_size(needed_size) < needed_size)
        return false; // the buffer resize request failed
    }

  // clear target memory
  memset(mem, 0x00, mem_octets);

  if (offset%8 && (offset+extract)%8 && offset/8==(offset+extract)/8)
    {
      uint8_t mask, value;
      mask = masks[LOWER_BITS(offset)] ^ masks[LOWER_BITS(offset+extract)];
      value = *(m_pdata+offset/8) & mask;
      value >>= LOWER_BITS(offset+extract);
      *((uint8_t*)mem+mem_octets-1) = value;
    }
  else
    {

      // first whole octet to transfer is integer divisor plus 1 for any remainder
      fwo = OCTET_CEIL(offset);
      // calculate last whole octet
      lwo = OCTET_CEIL(offset+extract);
      // calculate octets to transfer
      ot = lwo - fwo;
      // calculate actual destination within memory area
      ad = (uint8_t*)mem + mem_octets - ot;
      // and perform memory copy (if we have anything to copy)
      // if we only have a first partial, then there won't be a copy here
      // and it will occur later when we patch up the front
      if (ot)
        memcpy(ad, m_pdata+fwo, ot);

      // downward shift on chunk
      bsd = LOWER_BITS(offset+extract);
      if (bsd)
        {
          mask = masks[bsd];
          // calculate bits to shift upward in each chunk
          bsu = 8 - bsd;
          for (p = ad+ot-1; p > ad; p--)
            {
              *p >>= bsd;
              temp = *(p-1) & mask;
              temp <<= bsu;
              *p |= temp;
            }
          // don't forget the uppermost byte which only needs to be shifted
          *p >>= bsd;
        }

      befpo = LOWER_BITS(offset);
      if (befpo)
        {
          po = offset / 8;
          if (bsd)
            {
              // error somewhere in here
              temp = *( m_pdata + po );
              temp &= masks[befpo];
              temp <<= bsd;
              *(ad+fwo-1) |= temp;
            }
          if (befpo > bsd)
            {
              temp = *(m_pdata+po);
              temp &= masks[befpo];
              temp >>= bsd;
              *(ad-1) = temp;
            }
        }
    }
  return true;
}

bool Buffer::pack(const void* mem, size_t mem_octets, size_t offset, size_t destsize, size_t n)
{
  uint8_t *pbuf, *pmem;
  uint8_t temp, temp1, temp2, mask;
  size_t copy;
  uint8_t *pstart, *pend, *ptemp;
  size_t start, end, lb, ub, sb, eb, rb, bits;
  size_t needed_size;

  // are we extracting from a null address?
  if ( mem == NULL )
    return false;

  // are we being asked to pack in more bits than will fit?
  if (n > destsize)
    n = destsize;

  // check to see if the request is beyond the current size
  if ( m_size < OCTET_CEIL(offset+destsize))
    {
      if ( !m_dynamic )
        return false; // there's nothing we can do if we aren't dynamic

      // try to adjust the buffer size to accomodate the request
      needed_size = OCTET_CEIL(offset+destsize);
      if (set_size(needed_size) < needed_size)
        return false; // the buffer resize request failed
    }

  start = offset + destsize - n;
  end = offset + destsize;

  // does buffer target end on octet boundary?
  if ( end%8 == 0)
    { // ends on octet boundary
      // copy as much as possible
      copy = n/8; // calculate whole octets to copy
      pbuf = m_pdata + end/8 - copy; // calculate buffer location by moving forward to end and backing up # octets to copy
      pmem = (uint8_t*)mem + mem_octets - copy; // calculate copy location in memory similar to pbuf
      memcpy(pbuf, pmem, copy);

      // copy any remaining bits
      if (n%8)
        { // check to see if there are remaining bits
          pbuf--;
          pmem--;
          temp1 = *(pmem);
          temp1 &= masks[n%8];
          temp2 = *(pbuf);
          temp2 &= ~masks[n%8];
          *(pbuf) = temp1 | temp2;
        }

    }
  // check to see if we have a partial octet, which we have if start and end are in the same octet
  else if (start/8 == end/8)
    {
      bits = (end-start)%8; // calculate how many bits we will extract
      mask = masks[bits]; // get the bit mask to use
      lb = LOWER_BITS(end); // calculate the shift offset
      temp = *((uint8_t*)mem+mem_octets-1); // get last octet of memory
      temp <<= lb; // shift up into position
      mask <<= lb; // shift mask up into position as well
      temp &= mask; // mask out all bits not copied
      *(m_pdata+start/8) &= ~mask; // mask out the bits to be copied in destination
      *(m_pdata+start/8) |= temp; // and finally bring in our current bits
    }
  else
    { // we don't end on an octet boundary and cross at least one octet boundary
      // jump to the last octet
      pmem = (uint8_t*)mem + mem_octets - 1;
      sb = UPPER_BITS(end); // calculate the number of shift bits to keep
      lb = 8 - sb; // calculate the number of lower bits to drop
      pbuf = m_pdata + end/8;
      *pbuf &= masks[lb]; // clear upper bits
      *pbuf |= *pmem << lb; // bring in upper bits
      pbuf--; // start with pbuf at first whole octet, but memory still at first octet
      while (pbuf > m_pdata + start/8)
        {
          *pbuf = *pmem >> sb; // bring in upper bits of memory into lower bits of buffer
          pmem--; // move back one octet on memory
          *pbuf |= *pmem << lb; // and now lower bits of memory into upper bits of buffer
          pbuf--; // and back up one octet in the buffer
        }
      // we just finished the last whole octet, now we need to do the leading octet
        if (start%8 == 0) {
          *pbuf = *pmem >> sb; // bring in upper bits of memory into lower bits of buffer
          pmem--; // move back one octet on memory
          *pbuf |= *pmem << lb; // and now lower bits of memory into upper bits of buffer
        }
        else
        { // check to see if there is any work to be done
          rb = LOWER_BITS(start); // remaining bits for leading octet
          // use as many bits as we can from the current memory pointer
          mask = masks[((rb>lb)?lb:rb)]; // get mask which is lesser of lb or sb
          temp = *pbuf;
          *pbuf &= ~mask; // clear bits for lesser of rb or lb
          temp = *pbuf;
          *pbuf |= (*pmem >> sb) & mask; // get bits shifted down and potentially clear for rb mask
          temp = *pbuf;
          if (rb > lb)
            { // rb was the greater so we need to get rb-lb more bits from memory
              eb = rb - lb;
              pmem--; // move back one more octet
              mask = mask ^ masks[rb]; // get mask which represents only the extra bits
              temp = *pbuf;
              *pbuf &= ~mask; // clear the extra bits
              temp = *pbuf;
              temp = *pmem;
              *pbuf |= (*pmem << lb) & mask; // shift memory up and mask only needed bits
              temp = *pbuf;
            }
        }
    }

  // zero out any leading data in buffer
  if (destsize > n)
    clear_bits(offset, destsize-n, true);

  on_data_changed();
  return true;
}

void Buffer::set_data(const void* data, size_t external_size)
{
  if (m_dynamic)
    set_size(external_size);
  else
    external_size = std::min(external_size, m_size);

  memcpy(m_pdata, data, m_size);
  if (external_size < m_size)
    memset(m_pdata+external_size, 0x00, m_size-external_size);
  on_data_changed();
}

void Buffer::clear()
{
  memset(m_pdata, 0x00, m_size);
  on_data_changed();
}

bool Buffer::clear_bits(size_t offset, size_t bits, bool suppress)
{
  if (m_pdata == NULL || OCTET_CEIL(offset+bits) > m_size)
    return false;
  if (offset%8 && (offset+bits)%8 && offset/8==(offset+bits)/8)
    {
      uint8_t mask;
      mask = masks[LOWER_BITS(offset)] ^ masks[LOWER_BITS(offset+bits)];
      *(m_pdata+offset/8) &= ~mask;
    }
  else
    {
      // clear the leading portion (if any)
      if ( LOWER_BITS(offset) )
        *( OFFSET_PTR_FLOOR(offset) ) &= ~(masks[LOWER_BITS(offset)]);
      // clear any whole octets (if any)
      if ( OFFSET_PTR_CEIL(offset) != OFFSET_PTR_FLOOR(offset+bits) )
        memset(OFFSET_PTR_CEIL(offset), 0x00, OFFSET_PTR_FLOOR(offset+bits)-OFFSET_PTR_CEIL(offset));
      // clear the trailing portion (if any)
      if ( UPPER_BITS(offset+bits) )
        *( OFFSET_PTR_FLOOR(offset+bits) ) &= masks[LOWER_BITS(offset+bits)];
    }

  if (!suppress)
    on_data_changed();

  return true;
}

Buffer& Buffer::operator=( const Buffer & other )
{
  // the dynamic and owner conditions will always be the same as the other buffer
  m_dynamic = other.m_dynamic;
  m_sizemax = other.m_sizemax;

  // duplicate buffer
  set_size(other.m_size);
  memcpy(m_pdata, other.m_pdata, m_size);
  on_data_changed();

  return *this;
}

size_t Buffer::set_size( size_t size_request )
{
  uint8_t* new_buffer;
  bool location_changed = false;

  if (!m_dynamic)
    return m_size;

  // a request of size 0 is a request to free the buffer
  if (size_request == 0)
    {
      free_buffer();
      on_data_location_changed();
      on_size_changed();
      return m_size;
    }

  // make sure we don't make it larger than the max size
  // if a maximum size is set
  if (m_sizemax > 0)
    size_request = std::min(size_request, m_sizemax);

  new_buffer = (uint8_t*) realloc(m_pdata, size_request);
  // check to make sure the allocation went through
  if (new_buffer != NULL)
    {
      // check to see if the location changed, but if so, wait until m_pdata and
      // m_size are changed before sending notifications
      if (m_pdata != new_buffer)
        location_changed = true;
      m_pdata = new_buffer;
      // clear any new memory if there was any
      if (size_request > m_size)
        memset(m_pdata+m_size, 0x00, size_request-m_size);
      m_size = size_request;
      on_data_location_changed();
      on_size_changed();
    }

  return m_size;
}

void Buffer::free_buffer( )
{
  if (m_pdata != NULL)
    free(m_pdata);
  m_pdata = NULL;
  m_size = 0;
}

size_t Buffer::get_sizemax( )
{
  return m_sizemax;
}

void Buffer::set_sizemax(size_t sizemax)
{
  m_sizemax = sizemax;
  if (m_sizemax < m_size)
    {
      set_size(m_sizemax);
    }
}

void bit::Buffer::on_data_location_changed( )
{
  m_signal_data_location_changed();
}

void bit::Buffer::on_size_changed( )
{
  m_signal_size_changed();
}

void bit::Buffer::on_data_changed( )
{
  if (m_signal_data_changed_emitting)
    {
      m_signal_data_changed_need_reemit = true;
      return;
    }
  m_signal_data_changed_emitting = true;
  do
    {
      m_signal_data_changed();
    }
    while (m_signal_data_changed_need_reemit);
  m_signal_data_changed_need_reemit = false;
  m_signal_data_changed_emitting = false;
  return;
}

conexus::IO& operator<<(conexus::IO& io, bit::Buffer& bp) {
  io.write(bp.data(), bp.size());
  return io;
}

conexus::IO& operator<<(conexus::IO& io, bit::Data& d) {
  io.write(d.data.get(), d.size);
  return io;
}

conexus::IO& operator>>(conexus::IO& io, bit::Buffer& bp) {
  conexus::Data d = io.read();
  bp.set_data(d.data.get(), d.size);
  return io;
}

conexus::IO& operator>>(conexus::IO& io, bit::Data& bit_data) {
  conexus::Data conexus_data = io.read();
  bit_data.data = conexus_data.data;
  bit_data.size = conexus_data.size;
  return io;
}

