
/* This file is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option) */
/* any later version. */

/* This file is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with GNU Emacs; see the file COPYING.  If not, write to */
/* the Free Software Foundation, Inc., 59 Temple Place - Suite 330, */
/* Boston, MA 02111-1307, USA. */

/* Copyright (C) 2004 California Digital Corporation */
/* $Id: regexp.c,v 1.7 2004/04/29 22:51:20 summerisle Exp $ */


#ifdef HAVE_ERROR_H
#include <error.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <regex.h>
#include <errno.h>

#include "regexp.h"

/* handle_regerror - do the dance necessitated by ugly POSIX API */
/* errcode = REG_* integer error code  */
/* compiled = the compiled regexp that caused the error */

void
handle_regerror (int errcode, regex_t *compiled)
{
  size_t length;
  char *buffer;

  length = regerror (errcode, compiled, 0, 0);
  buffer = malloc (length);
  if (!buffer) error (1, errno, "handle_regerror");
  (void)regerror (errcode, compiled, buffer, length);
  error (1, 0, "%s", buffer);
}

/* matches either a blank line or a comment line */
static const char blank_or_comment_pattern[] =
   /* a blank line */
   "(^[[:space:]]+$)"

   "|" /* or ... */

   /* a line consisting of: optional whitespace followed by */
   "(^[[:space:]]*"
   /* a '#' character and optionally, additional characters */
   "#.*$)";

/* matches an assignment line */
static const char assignment_pattern[] =
   /* optional whitespace */
   "^[[:space:]]*"
   /* followed by a parameter name */
   "([-_[:alnum:]]+)"
   /* followed by optional whitespace */
   "[[:space:]]*"
   /* followed by an '=' character */
   "="
   /* followed by optional whitespace */
   "[[:space:]]*"
   /* followed by a value */
   "([-_[:alnum:]]+)"
   /* followed by optional whitespace */
   "[[:space:]]*$";

regex_t blank_or_comment_regex;

regex_t assignment_regex;

struct rxpair
{
  const char* pattern;
  regex_t* regex;
};

static struct rxpair patterns[] =
  {
    { blank_or_comment_pattern, &blank_or_comment_regex },
    { assignment_pattern, &assignment_regex },
    { 0, 0 }
  };

/* regexp_init - compile all the regular expressions used by this program */

void 
regexp_init (void)
{
  int i;

  for (i = 0; patterns[i].pattern; i++)
    {
      int status;

      status = regcomp (patterns[i].regex, patterns[i].pattern, REG_EXTENDED|REG_NEWLINE);
      if (status != 0) handle_regerror (status, patterns[i].regex);
    }
}
