#! /usr/bin/env perl

# This file is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.

# This file is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with GNU Emacs; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

# Copyright (C) 2004 California Digital Corporation

# $Id: options.pl,v 1.6 2004/08/12 20:30:49 summerisle Exp $

use 5.008_003;
use strict;
use warnings;

use constant {
    NAME => 0,
    KEY => 1,
    ARG => 2,
    ARG_OPTIONAL => 3,
    DOC => 4,
    GROUP => 5,
    HIDDEN => 6,
};

use FileHandle;

my @options =
    (
     [ "read", 'r', "NAME", 0, "Show name and value of parameter given by NAME", 1, 0 ],
     [ "no-name", 'n', undef, 0, "With -r: omit name, show only value", 1 , 0 ],
     [ "write", 'w', "'NAME=VALUE'", 0, "Assign VALUE to parameter given by NAME", 1 , 0 ],
     [ "to-file", 't', "FILE", 0, "Write names and values of all parameters to FILE", 2 , 0 ],
     [ "from-file", 'f', "FILE", 0, "Read assignments to parameters from FILE", 2 , 0 ],
     [ "dump", 'd', "FILE", 0, "Binary dump the entire CMOS into FILE", 3 , 0 ],
     [ "undump", 'u', "FILE", 0, "Read a binary dump of the entire CMOS from FILE", 3 , 0 ],
     [ "enumerate", 'e', "NAME", 0, "Show possible values of parameter NAME", 4 , 0 ],
     [ "config", 'C', "FILE", 0, "layout configuration in FILE [probe SMBIOS]", 4 , 0 ],
     [ "probe", 'P', undef, 0, "Probe for version information about the host BIOS", 4 , 0 ],
     [ "hash", 'H', undef, 0, "List MD5 hashes of BIOS versions covered by configuration file", 0 , 1 ],
     [ "directory", 'D', undef, 0, "Print configured directory for token map files", 0 , 1 ],
     [ "force", 'F', undef, 0, "Force use of config file not covering host BIOS", 4 , 0 ],
     );

my @vdocs =
    (
     "biosconfig reads or writes fields in the CMOS non-volatile store,",
     "most of which are associated with various functions of the PC",
     "Basic Input/Output System (BIOS).  The fields, their types and",
     "value ranges are specified in a configuration file in the style",
     "of GNUstep.  This format is familiar to users of the WindowMaker",
     "window manager or applets written for it, and is explained",
     "in greater detail in the full Info manual.",
     );

my $summary = 'read and write CMOS BIOS paramaters';

our $fh;

### GENERATE options.c ###

$fh = FileHandle->new ('options.c','>') or die "options.c: $!";
$fh->print ( <<__OPTIONS_C_HEADER__ );

#include "options.h"

#if (defined(HAVE_ARGP_H) && defined(HAVE_ARGP_PARSE))
#include <argp.h>

static struct argp_option options_list[] =
  {
__OPTIONS_C_HEADER__

foreach my $option (@options) {
    my $flag_names;
    if ($option->[ARG_OPTIONAL] && $option->[HIDDEN]) { $flag_names = 'OPTION_ARG_OPTIONAL|OPTION_HIDDEN'; }
    elsif ($option->[ARG_OPTIONAL]) { $flag_names = 'OPTION_ARG_OPTIONAL'; }
    elsif ($option->[HIDDEN]) { $flag_names = 'OPTION_HIDDEN'; }
    else { $flag_names = '0'; }
    $fh->print ( <<__OPTIONS_LIST_LINE_ARG__ ), next if $option->[ARG];
    { "$option->[NAME]", '$option->[KEY]', "$option->[ARG]", $flag_names, "$option->[DOC]", $option->[GROUP] },
__OPTIONS_LIST_LINE_ARG__

    $fh->print ( <<__OPTIONS_LIST_LINE_NOARG__ );
    { "$option->[NAME]", '$option->[KEY]', 0, $flag_names, "$option->[DOC]", $option->[GROUP] },
__OPTIONS_LIST_LINE_NOARG__

}

$fh->print ( <<__OPTIONS_LIST_TAIL__ );
    { 0, 0, 0, 0, 0, 0 }
  };

__OPTIONS_LIST_TAIL__

$fh->print ( <<__OPTIONS_PARSER_HEADER__ );
struct argp options_parser =
  {
    options_list,
    parse_options,
    NULL,
__OPTIONS_PARSER_HEADER__

foreach my $vdoc (@vdocs) {
    $fh->print ( <<__OPTIONS_PARSER_LINE__ );
    "$vdoc""\\n"
__OPTIONS_PARSER_LINE__
}

$fh->print ( <<__OPTIONS_PARSER_TAIL__ );
    "\\v",
    NULL,
    NULL,
    NULL
  };

#else /* (defined(HAVE_ARGP_H) && defined(HAVE_ARGP_PARSE)) */

__OPTIONS_PARSER_TAIL__

$fh->print ( <<__OPTIONS_GETOPT_HEADER__ );
const char getopt_options_list[] = 
__OPTIONS_GETOPT_HEADER__

$fh->print ('  "');
foreach my $option (@options) {
    $fh->print ($option->[KEY]), next unless !$option->[ARG];
    $fh->print ($option->[KEY] . '::'), next if $option->[ARG_OPTIONAL];
    $fh->print ($option->[KEY] . ':');
}
$fh->print ('";' . "\n");

$fh->print ( <<__OPTIONS_C_TAIL__ );

#endif /* (defined(HAVE_ARGP_H) && defined(HAVE_ARGP_PARSE)) */
__OPTIONS_C_TAIL__

$fh->close;

### GENERATE options.c DONE ###

### GENERATE biosconfig.pod ###

sub podline {
    my $directive = shift;
    $fh->print ("=$directive ");
    foreach my $item (@_) { $fh->print ("$item "); }
    $fh->print ("\n\n");
}

sub optcmp {
    return -1 if $a->[GROUP] >= 0 && $b->[GROUP] < 0;
    return 1 if $a->[GROUP] < 0 && $b->[GROUP] >= 0;
    return ($a->[GROUP] <=> $b->[GROUP] or $a->[NAME] cmp $b->[NAME]);
}

$fh = FileHandle->new ('biosconfig.pod', '>')
    or die "biosconfig.pod: $!";

&podline ("head1", "NAME");

$fh->print ( <<__BIOSCONFIG_POD_NAME__ );
biosconfig \- $summary

__BIOSCONFIG_POD_NAME__

&podline ("head1", "SYNOPSIS");

$fh->print ( <<__BIOSCONFIG_POD_SYNOPSIS__ );
B<biosconfig> [OPTIONS...]

__BIOSCONFIG_POD_SYNOPSIS__

&podline ("head1", "DESCRIPTION");
foreach my $vdoc (@vdocs) { $fh->print ("$vdoc\n"); }

$fh->print ("\n");
&podline ("head1", "OPTIONS");
&podline ("over", "2");

 PODOPT:
    foreach my $option (sort optcmp @options) {
        next PODOPT if $option->[HIDDEN];
        if (!$option->[ARG]) {
            &podline ("item", "B<-$option->[KEY],", "--$option->[NAME]>");
        } elsif ($option->[ARG_OPTIONAL]) {
            &podline ("item", "B<-$option->[KEY],", "--$option->[NAME]", "[=$option->[ARG]]>");
        } else {
            &podline ("item", "B<-$option->[KEY],", "--$option->[NAME]=$option->[ARG]>");
        }
        $fh->print ("$option->[DOC]\n\n");
    }

&podline ("back");
&podline ("head1", "EXAMPLES");

$fh->print ( <<__BIOSCONFIG_POD_EXAMPLES__ );
B<Warning:> all the token information in these examples, including
in particular the CMOS parameter names, is completely made up and does not
refer to any real BIOS.  Unfortunately, the real information, which we
have used during testing, is under NDAs.  We hope that once biosconfig
becomes a popular tool its intended users, i.e. administrators of clusters
and server farms, will demand that their vendors open this information,
and the vendors will do so in their own self-interest.

Read the name and value of a single parameter:

  biosconfig -r hyperthread

Read only the value of a single parameter:

  biosconfig -n -r console_baud_rate

Modify the value of a single parameter:

  biosconfig -w 'hyperthread = disable'

Output names and values of all known parameters into params.txt:

  biosconfig -t params.txt

or

  biosconfig -t- > params.txt

Modify multiple parameters, reading values from params.txt (which could
have been created by a previous invocation of B<biosconfig -t>):

  biosconfig -f params.txt

or

  biosconfig -f- < params.txt

Create a binary dump of the CMOS in cmos.bin:

  biosconfig -d cmos.bin

Restore the CMOS from a binary dump file cmos.bin,
created by a previous invocation of B<biosconfig -d>:

  biosconfig -u cmos.bin

List all possible values of a parameter:

  biosconfig -e console_baud_rate

__BIOSCONFIG_POD_EXAMPLES__

$fh->close;

### GENERATE biosconfig.pod DONE ###

### GENERATE options.texi ###

$fh = FileHandle->new ('options.texi', '>')
    or die "options.texi: $!";

$fh->print ( <<'__OPTIONS_TEXI_SYNOPSIS' );
@verbatim
SYNOPSIS
@end verbatim

@example
biosconfig @var{option} @dots{}
@end example

@verbatim
OPTIONS
@end verbatim

__OPTIONS_TEXI_SYNOPSIS

sub texiline {
    my $directive = shift;
    $fh->print ("\@$directive ");
    foreach my $item (@_) { $fh->print ("$item "); }
    $fh->print ("\n");
}

&texiline ("table", '@code');

TEXIOPT:
foreach my $option (sort optcmp @options) {
        next TEXIOPT if $option->[HIDDEN];
        if (!$option->[ARG]) {
            &texiline ("item", "-$option->[KEY]");
            &texiline ("itemx", "--$option->[NAME]");
        } elsif ($option->[ARG_OPTIONAL]) {
            &texiline ("item", "-$option->[KEY] [$option->[ARG]]");
            &texiline ("itemx", "--$option->[NAME]\[=$option->[ARG]]");
        } else {
            &texiline ("item", "-$option->[KEY] $option->[ARG]");
            &texiline ("itemx", "--$option->[NAME]=$option->[ARG]");
        }
        $fh->print ("$option->[DOC]\n\n");
    }

&texiline ("end", "table");

$fh->close;

### GENERATE options.texi DONE ###
