
/* This file is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option) */
/* any later version. */

/* This file is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with GNU Emacs; see the file COPYING.  If not, write to */
/* the Free Software Foundation, Inc., 59 Temple Place - Suite 330, */
/* Boston, MA 02111-1307, USA. */

/* Copyright (C) 2004 California Digital Corporation */
/* $Id: object.c,v 1.10 2004/04/29 22:51:20 summerisle Exp $ */

#include "object.h"

#ifdef HAVE_ERROR_H
#include <error.h>
#endif

#include <stdio.h>
#include <search.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <inttypes.h>

static int
compare_idents (const void* vp1, const void* vp2)
{
  return (strcmp (((ident_t*)vp1)->name, ((ident_t*)vp2)->name));
}

/* maybe_insert_ident - lookup an identifier in a symbol table and insert it if not found */
/* tree = points to symbol table represented as a tree (compatible with tsearch) */
/* name = the new identifier as a string */
/* line = line number of occurence */
/* column = starting column number of occurence */
/* return : pointer to ident object, either found or newly inserted */

ident_t*
maybe_insert_ident (void** tree, const char* name, int line, int column)
{
  ident_t* pbait;
  ident_t** pfound;

  pbait = malloc (sizeof (ident_t));
  if (!pbait) error (1, errno, "maybe_insert_ident");
  pbait->name = strdup (name);
  if (!pbait->name) error (1, errno, "maybe_insert_ident");
  pbait->line = line;
  pbait->column = column;
  pfound = tsearch (pbait, tree, compare_idents);
  if (!pfound) error (1, errno, "maybe_insert_ident");
  else if (*pfound != pbait)
    {
      free (pbait->name);
      free (pbait);
    }
  return *pfound;
}

/* This function is over-long for a reason --- it is the real work of the program :) */

/* analyze_layout - check data structure returned from config file parser, fill backpatches */
/* lp = points to layout structure returned from parser */

void
analyze_layout (layout_t* lp)
{
  /* first, if there are no banks, create a default one out of thin air */
  if (!lp->banks)
    {
      ident_t* id;
      bank_t* b;

      id = maybe_insert_ident (&lp->tree, "bank0", 0, 0);
      b = malloc (sizeof (bank_t));
      if (!b) error (1, errno, "analyze_layout");
      b->name = id;
      b->address_port = 0x70;
      b->data_port = 0x71;
      b->length = 128;
      b->next = NULL;
      lp->banks = b;
    }
  /* check banks */
  {
    bank_t* p_bank;

    for (p_bank = lp->banks; p_bank != NULL; p_bank = p_bank->next)
      {
        if (p_bank->address_port == 0xffff || p_bank->data_port == 0xffff ||
            p_bank->length == -1 )
          error (1, 0, "bank %s lacks an attribute", p_bank->name->name);
      }
  }
  /* process checksum list */
  {
    checksum_t* p_checksum;

    for (p_checksum = lp->checksums; p_checksum != NULL; p_checksum = p_checksum->next)
      {
        if (p_checksum->start == -1 || p_checksum->type == ct_none)
          error (1, 0, "checksum %s lacks an attribute", p_checksum->name->name);
        {
          bank_t* p_bank;

          for (p_bank = lp->banks; p_bank != NULL; p_bank = p_bank->next)
            if (p_bank->name == p_checksum->bp_bank.u.name)
              {
                p_checksum->bp_bank.filled = 1;
                p_checksum->bp_bank.u.pbank = p_bank;
                break;
              }
          if (!p_checksum->bp_bank.filled)
            error (1, 0, "checksum %s refers to nonexistent bank %s",
                   p_checksum->name->name, p_checksum->bp_bank.u.name);
          if (p_checksum->start >= p_bank->length ||
              (p_checksum->type != single_byte && p_checksum->start >= p_bank->length - 1))
            error (1, 0, "checksum %s out of range for bank %s",
                   p_checksum->name->name, p_bank->name->name);
        }
      }
  }
  /* process region list */
  {
    region_t* p_region;

    for (p_region = lp->regions; p_region != NULL; p_region = p_region->next)
      {
        if (p_region->start_byte == -1 || p_region->byte_length == -1)
          error (1, 0, "region %s lacks an attribute", p_region->name->name);
        {
          bank_t* p_bank;

          for (p_bank = lp->banks; p_bank != NULL; p_bank = p_bank->next)
            if (p_bank->name == p_region->bp_bank.u.name)
              {
                p_region->bp_bank.filled = 1;
                p_region->bp_bank.u.pbank = p_bank;
                break;
              }
          if (!p_region->bp_bank.filled)
            error (1, 0, "region %s refers to nonexistent bank %s",
                   p_region->name->name, p_region->bp_bank.u.name);
          if (p_region->start_byte >= p_bank->length ||
              (p_region->start_byte + p_region->byte_length > p_bank->length))
            error (1, 0, "region %s out of range for bank %s",
                   p_region->name->name, p_bank->name->name);
        }
      }
  }
  /* process entry list */
  {
    cmentry_t* p_cmentry;

    for (p_cmentry = lp->entries; p_cmentry != NULL; p_cmentry = p_cmentry->next)
      {
        if (p_cmentry->start_bit == -1 || p_cmentry->bit_length == -1 ||
            p_cmentry->type == et_none || p_cmentry->bp_bank.u.name == NULL ||
            (p_cmentry->type == enumtype && p_cmentry->bp_enum.u.name == NULL))
          error (1, 0, "entry %s lacks an attribute", p_cmentry->name->name);
        {
          bank_t* p_bank;

          for (p_bank = lp->banks; p_bank != NULL; p_bank = p_bank->next)
            if (p_bank->name == p_cmentry->bp_bank.u.name)
              {
                p_cmentry->bp_bank.filled = 1;
                p_cmentry->bp_bank.u.pbank = p_bank;
                break;
              }
          if (!p_cmentry->bp_bank.filled)
            error (1, 0, "entry %s refers to nonexistent bank %s",
                   p_cmentry->name->name, p_cmentry->bp_bank.u.name);
          if (p_cmentry->start_bit / 8 >= p_bank->length ||
              (p_cmentry->start_bit + p_cmentry->bit_length) / 8 > p_bank->length)
            error (1, 0, "entry %s out of range for bank %s",
                   p_cmentry->name->name, p_bank->name->name);
        }
        if (p_cmentry->type == enumtype && p_cmentry->bp_enum.u.name)
          {
            enumeration_t* p_enum;

            for (p_enum = lp->enums; p_enum != NULL; p_enum = p_enum->next)
              if (p_enum->name == p_cmentry->bp_enum.u.name)
                {
                  p_cmentry->bp_enum.filled = 1;
                  p_cmentry->bp_enum.u.penum = p_enum;
                  break;
                }
            if (!p_cmentry->bp_enum.filled)
              error (1, 0, "entry %s refers to nonexistent enumeration %s",
                     p_cmentry->name->name, p_cmentry->bp_enum.u.name);
          }
        if (p_cmentry->pbp_checksums != NULL)
          {
            backpatch_t* pbp;

            for (pbp = p_cmentry->pbp_checksums; pbp != NULL; pbp = pbp->next)
              {
                checksum_t* pchk;

                for (pchk = lp->checksums; pchk != NULL; pchk = pchk->next)
                  if (pchk->name == pbp->u.name)
                    {
                      pbp->filled = 1;
                      pbp->u.pchecksum = pchk;
                      break;
                    }
                if (!pbp->filled)
                  error (1, 0, "entry %s refers to nonexistent checksum %s",
                         p_cmentry->name->name, pbp->u.name);
              }
          }
      }
  }
}

/* find_entry - locate data structure for a layout entry based on its name */
/* name = name of entry */
/* lp = points to layout data structure returned from parser */
/* return : data structure for layout entry */

cmentry_t*
find_entry (char* name, layout_t* lp)
{
  cmentry_t* pe;
  ident_t bait;
  ident_t** pfound;

  bait.name = name;
  pfound = tfind (&bait, &lp->tree, compare_idents);
  if (!pfound) error (1, 0, "find_entry: no entry %s", name);
  for (pe = lp->entries; pe != NULL; pe = pe->next)
    if (pe->name == *pfound) break;
  if (!pe) error (1, 0, "find_entry: no entry %s", name);
  return pe;
}

/* lookup_value - translate a string, either a literal number or an enum, into a value */
/* pe = points to layout entry for which the value is intended */
/* val_name = string representation of value */
/* lp = points to layout structure returned from parser */
/* return : numeric value */

uint32_t
lookup_value (cmentry_t* pe, char* val_name, layout_t* lp)
{
  switch (pe->type)
    {
    case numeric:
      {
        char* endptr;
        unsigned long ul;

        errno = 0;
        ul = strtoul (val_name, &endptr, 0);
        /* ul > UINT32_MAX could happen on a 64 bit architecture. */
        if (errno == ERANGE || *endptr != '\0' || ul > UINT32_MAX)
          error (1, errno, "invalid value %s for %s", val_name, pe->name->name);
        return (uint32_t)ul;
      }
    case enumtype:
      {
        enumeration_t* penum;
        enum_value_t* peval;
        ident_t bait;
        ident_t** pfound;

        penum = pe->bp_enum.u.penum;
        bait.name = val_name;
        pfound = tfind (&bait, &lp->tree, compare_idents);
        if (!pfound) error (1, 0, "lookup_value: no enumeration %s", val_name);
        for (peval = penum->values; peval != NULL; peval = peval->next)
          if (peval->name == *pfound) break;
        if (!peval) error (1, 0, "lookup_value: %s is not an enumeration of kind %s",
                           val_name, penum->name->name);
        return peval->value;
      }
    case reserved:
      error (1, 0, "lookup_value: entry %s is reserved", pe->name->name);
    default:
      error (2, 0, "lookup_value: can't happen (internal error, please report)");
    }
  return 0;                     /* not reached */
}

/* print_value - translate a value, either a number or an enum, into a string and output it */
/* fp = file handle to output value on */
/* pe = points to layout entry for which the value is intended */
/* val = numeric representation of value */
/* lp = points to layout structure returned from parser */

void
print_value (FILE* fp, cmentry_t* pe, uint32_t val, layout_t* lp)
{
  switch (pe->type)
    {
    case numeric:
    case reserved:
      fprintf (fp, "%u", val);
      break;
      
    case enumtype:
      {
        enumeration_t* penum;
        enum_value_t* peval;

        penum = pe->bp_enum.u.penum;
        for (peval = penum->values; peval != NULL; peval = peval->next)
          if (peval->value == val) break;
        if (!peval) fprintf (fp, "UNKNOWN%u", val);
	else fprintf (fp, "%s", peval->name->name);
	break;
      }      
    default:
      error (2, 0, "print_value: can't happen (internal error, please report)");
    }
}
