
/* This file is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option) */
/* any later version. */

/* This file is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with GNU Emacs; see the file COPYING.  If not, write to */
/* the Free Software Foundation, Inc., 59 Temple Place - Suite 330, */
/* Boston, MA 02111-1307, USA. */

/* Copyright (C) 2004 Ian Zimmerman */

/* $Id: escape.c,v 1.3 2004/04/29 22:51:19 summerisle Exp $ */

#include "escape.h"

#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>

static const unsigned char codes[] = "a\ab\bf\fn\nr\rt\tv\v";

/* unescape_char - resolve a c-style escape sequence */
/* src = address of source string, updated to point past converted char */
/* endsrc = pointer to end of string being scanned */
/* return : converted character value, or -1 on error */

int
unescape_char (const char** src, const char* endsrc)
{
  const unsigned char* p;
  const unsigned char* q;

  p = *src;
  if ((const char*)p >= endsrc) return -1;
  if (*p != '\\')
    {
      *src = p + 1;
      return (int)((unsigned)*p);
    }
  if ((const char*)++p >= endsrc) return -1;
  q = strchr (codes, *p);
  if (q != NULL)
    {
      *src = p + 1;
      return (int)((unsigned)(q[1]));
    }
  if (*p == '0' || *p == '1')
    {
      int i;
      int ret;

      if (*p == '0' && ((const char*)(p + 1) >= endsrc ||
                        p[1] < '0' || p[1] > '7'))
        {
          *src = p + 1;
          return 0;
        }
      if ((const char*)(p + 2) >= endsrc) return -1;
      ret = *p - '0';
      for (i = 1; i <= 2; i++)
        {
          if (p[i] < '0' || p[i] > '7') return -1;
          ret = (ret * 8) + (p[i] - '0');
        }
      *src = p + 3;
      return ret;
    }
  *src = p + 1;
  return (int)((unsigned)*p);
}

/* unescape_array - resolve an array of c-style escape sequences */
/* dst = destination string (where to write resolved characters) */
/* src = source array */
/* len = size of source array */
/* return : pointer to end of converted string in dst, or NULL on error */

char*
unescape_array (char* dst, const char* src, size_t len)
{
  const char* psrc;
  const char* endsrc;
  char* pdst;
  int c;

  psrc = src;
  endsrc = src + len;
  pdst = dst;
  while (psrc < endsrc)
    {
      c = unescape_char (&psrc, endsrc);
      if (c < 0) return NULL;
      *pdst++ = (char) c;
    }
  *pdst = '\0';
  return pdst;
}

/* file_unescape_array - write array of c-style escape sequences to file */
/* fp = file where to write resolved characters */
/* src = source array */
/* len = size of source array */
/* return : number of characters converted (and written) or -1 on error */

ssize_t
file_unescape_array (FILE* fp, const char* src, size_t len)
{
  const char* psrc;
  const char* endsrc;
  int c;
  ssize_t i;

  psrc = src;
  endsrc = src + len;
  i = 0;
  while (psrc < endsrc)
    {
      c = unescape_char (&psrc, endsrc);
      if (c < 0) return -1;
      if (fputc (c, fp) < 0) return -1;
      i++;
    }
  return i;
}

/* unescape_string - resolve a string of c-style escape sequences */
/* dst = destination string (where to write resolved characters) */
/* src = source string */
/* return : pointer to end of converted string in dst, or NULL on error */

char*
unescape_string (char* dst, const char* src)
{
  return unescape_array (dst, src, strlen (src));
}

/* file_unescape_string - write string of c-style escape sequences to file */
/* fp = file where to write resolved characters */
/* src = source string */
/* return : number of characters converted (and written) or -1 on error */

ssize_t
file_unescape_string (FILE* fp, const char* src)
{
  return file_unescape_array (fp, src, strlen (src));
}

/* escape_char - convert a character value to a c-style escape sequence */
/* dst = where to put escape sequence */
/* c = source character value */
/* extras = list of additional printable characters to escape (", ', etc.) */
/* return : pointer to end of escape string in dst */

char*
escape_char (char* dst, int c, const char* extras)
{
  const char* q;

  if (c == '\\' || (extras != NULL && strchr (extras, c) != NULL))
    {
      dst [0] = '\\';
      dst [1] = (char) c;
      dst [2] = '\0';
      return (dst + 2);
    }
  if (isprint (c))
    {
      dst [0] = (char) c;
      dst [1] = '\0';
      return (dst + 1);
    }
  q = strchr (codes, c);
  if (q != NULL)
    {
      dst [0] = '\\';
      dst [1] = *(q - 1);
      dst [2] = '\0';
      return (dst + 2);
    }
  dst [0] = '\\';
  sprintf (dst + 1, "%03o", (unsigned)c);
  dst [4] = '\0';
  return (dst + 4);
}

/* escape_array - convert an array of characters to c-style escape sequences */
/* dst = where to put escape sequences */
/* src = source character array */
/* len = length of source character array */
/* extras = list of additional printable characters to escape (", ', etc.) */
/* return : pointer to end of escape string in dst */

char*
escape_array (char* dst, const char* src, size_t len, const char* extras)
{
  const char* psrc;
  const char* endsrc;
  char* pdst;

  psrc = src;
  endsrc = src + len;
  pdst = dst;
  while (psrc < endsrc)
    pdst = escape_char (pdst, *psrc++, extras);
  *pdst = '\0';
  return pdst;
}

/* file_escape_array - write array of characters to file as escape sequences */
/* fp = file where to write escape sequences */
/* src = source character array */
/* len = length of source character array */
/* extras = list of additional printable characters to escape (", ', etc.) */
/* return : number of characters (not sequences) written, or -1 on error */

ssize_t
file_escape_array (FILE* fp, const char* src, size_t len, const char* extras)
{
  const char* psrc;
  const char* endsrc;
  char buf[5];
  ssize_t i;

  psrc = src;
  endsrc = src + len;
  i = 0;
  while (psrc < endsrc)
    {
      escape_char (buf, *psrc++, extras);
      if (fputs (buf, fp) < 0) return -1;
      i += strlen (buf);
    }
  return i;
}

/* escape_string - convert a string of characters to escape sequences */
/* dst = where to put escape sequences */
/* src = source character string */
/* extras = list of additional printable characters to escape (", ', etc.) */
/* return : pointer to end of escape string in dst */

char*
escape_string (char* dst, const char* src, const char* extras)
{
  return escape_array (dst, src, strlen (src), extras);
}

/* file_escape_string - write string of characters to file as escape sequences */
/* fp = file where to write escape sequences */
/* src = source character string */
/* extras = list of additional printable characters to escape (", ', etc.) */
/* return : number of characters (not sequences) written, or -1 on error */

ssize_t
file_escape_string (FILE* fp, const char* src, const char* extras)
{
  return file_escape_array (fp, src, strlen (src), extras);
}
