
/* This file is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option) */
/* any later version. */

/* This file is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with GNU Emacs; see the file COPYING.  If not, write to */
/* the Free Software Foundation, Inc., 59 Temple Place - Suite 330, */
/* Boston, MA 02111-1307, USA. */

/* Copyright (C) 2004 California Digital Corporation */
/* $Id: cmosop.c,v 1.10 2004/10/11 17:44:19 itz Exp $ */

#include "cmosop.h"
#include "cmoslow.h"

#ifdef HAVE_ERROR_H
#include <error.h>
#endif

#include <sys/io.h>
#include <errno.h>
#include <inttypes.h>

static inline int
safe_max (int i, int j)
{
  return (i > j ? i : j);
}

static inline int
safe_min (int i, int j)
{
  return (i < j ? i : j);
}

/* cmos_set_privs - turn on or off privileges necessary to access a pair of ports */
/* address_port = first port to change privileges for */
/* data_port = second port to change privileges for */
/* on = turn on or off */

static void
cmos_set_privs (uint16_t address_port, uint16_t data_port, int on)
{
  if (ioperm (address_port, 1, on) != 0)
    error (1, errno, "ioperm (%x, 1, %d)", address_port, on);
  if (ioperm (data_port, 1, on) != 0)
    error (1, errno, "ioperm (%x, 1, %d)", data_port, on);
}

/* cmos_set_checksum_privs - turn on or off privileges to update a list of checksums */
/* csums = list of checksums wrapped in backpatch structures */
/* on = turn on or off */

static void
cmos_set_checksum_privs (backpatch_t* csums, int on)
{
  backpatch_t* pbp;

  for (pbp = csums; pbp != NULL; pbp = pbp->next)
    {
      checksum_t* pchk;
      bank_t* pchkbank;

      pchk = pbp->u.pchecksum;
      pchkbank = pchk->bp_bank.u.pbank;
      cmos_set_privs (pchkbank->address_port, pchkbank->data_port, on);
    }
}

/* cmos_modify_byte - update a byte value in the CMOS with a masked value */
/* addr_port = typically  0x70 */
/* data_port = typically 0x71 */
/* byte = offset in CMOS to the updated byte */
/* start_bit = low order bit of updated field */
/* mask = bit mask to extract field after shifting right to bit 0 */
/* value = field defined by start_bit and mask receives this value */
/* return : delta between new and old byte value */

static int
cmos_modify_byte (uint16_t addr_port, uint16_t data_port, int byte,
                  int start_bit, uint8_t mask, uint32_t value)
{
  uint8_t old_byte;
  uint8_t new_byte;

  old_byte = cmos_read_byte (addr_port, data_port, byte);
  new_byte = (old_byte & ~(mask << start_bit)) | ((value & mask) << start_bit);
  cmos_write_byte (addr_port, data_port, byte, new_byte);
  return ((int)new_byte & 0xff) - ((int)old_byte & 0xff);
}

/* cmos_fake_modify_byte - fake update a byte value in the CMOS with a masked value */
/* addr_port = typically  0x70 */
/* data_port = typically 0x71 */
/* byte = offset in CMOS to the updated byte */
/* start_bit = low order bit of updated field */
/* mask = bit mask to extract field after shifting right to bit 0 */
/* value = field defined by start_bit and mask receives this value */
/* return : delta between new and old byte value */

static int
cmos_fake_modify_byte (uint16_t addr_port, uint16_t data_port, int byte,
                       int start_bit, uint8_t mask, uint32_t value)
{
  uint8_t old_byte;
  uint8_t new_byte;

  old_byte = cmos_read_byte (addr_port, data_port, byte);
  new_byte = (old_byte & ~(mask << start_bit)) | ((value & mask) << start_bit);
  cmos_fake_write_byte (addr_port, data_port, byte, new_byte);
  return ((int)new_byte & 0xff) - ((int)old_byte & 0xff);
}

/* cmos_extract_byte - extract a field from a byte in the CMOS */
/* addr_port = typically 0x70 */
/* data_port = typically 0x71 */
/* byte = offset in CMOS to the byte to extract from */
/* start_bit = low order bit of extracted field */
/* mask = bit mask to extract field after shifting right to bit 0 */
/* return : value of extracted field defined by start_bit and mask */

static uint32_t
cmos_extract_byte (uint16_t addr_port, uint16_t data_port,
                   int byte, int start_bit, uint8_t mask)
{
  uint8_t val;

  val = cmos_read_byte (addr_port, data_port, byte);
  return ((uint32_t)((val >> start_bit) & mask) & 0xff);
}

/* cmos_update_checksum - update a single checksum */
/* csum = points to checksum to update */
/* delta = change in bytes covered by csum, to be added into csum */

static void
cmos_update_checksum (checksum_t* csum, int delta)
{
  bank_t* bank;

  bank = csum->bp_bank.u.pbank;  
  switch (csum->type)
    {
    case single_byte:
      {
        uint8_t old_csum;

        old_csum = cmos_read_byte (bank->address_port, bank->data_port, csum->start);
        cmos_write_byte (bank->address_port, bank->data_port, csum->start,
                         (uint8_t)(((int)old_csum & 0xff) + delta));
      }
      break;

    case low_first:
      {
        uint16_t old_csum;
        uint8_t old_csum_low;
        uint8_t old_csum_high;
        uint16_t new_csum;
        uint8_t new_csum_low;
        uint8_t new_csum_high;

        old_csum_low = cmos_read_byte (bank->address_port, bank->data_port, csum->start);
        old_csum_high = cmos_read_byte (bank->address_port, bank->data_port, csum->start + 1);
        old_csum = ((uint16_t)old_csum_high << 8) | old_csum_low;
        new_csum = (uint16_t)(((int)old_csum & 0xffff) + delta);
        new_csum_low = (uint8_t)(new_csum & 0xff);
        new_csum_high = (uint8_t)(new_csum >> 8);
        cmos_write_byte (bank->address_port, bank->data_port, csum->start, new_csum_low);
        cmos_write_byte (bank->address_port, bank->data_port, csum->start + 1, new_csum_high);
      }
      break;

    case high_first:
      {
        uint16_t old_csum;
        uint8_t old_csum_low;
        uint8_t old_csum_high;
        uint16_t new_csum;
        uint8_t new_csum_low;
        uint8_t new_csum_high;

        old_csum_low = cmos_read_byte (bank->address_port, bank->data_port, csum->start + 1);
        old_csum_high = cmos_read_byte (bank->address_port, bank->data_port, csum->start);
        old_csum = ((uint16_t)old_csum_high << 8) | old_csum_low;
        new_csum = (uint16_t)(((int)old_csum & 0xffff) + delta);
        new_csum_low = (uint8_t)(new_csum & 0xff);
        new_csum_high = (uint8_t)(new_csum >> 8);
        cmos_write_byte (bank->address_port, bank->data_port, csum->start + 1, new_csum_low);
        cmos_write_byte (bank->address_port, bank->data_port, csum->start, new_csum_high);
      }
      break;

    default:
      error (2, 0, "cmos_update_checksum: can't happen (internal error, please report)");
    }
}

/* cmos_fake_update_checksum - fake update a single checksum */
/* csum = points to checksum to update */
/* delta = change in bytes covered by csum, to be added into csum */

static void
cmos_fake_update_checksum (checksum_t* csum, int delta)
{
  bank_t* bank;

  bank = csum->bp_bank.u.pbank;  
  switch (csum->type)
    {
    case single_byte:
      {
        uint8_t old_csum;

        old_csum = cmos_read_byte (bank->address_port, bank->data_port, csum->start);
        cmos_fake_write_byte (bank->address_port, bank->data_port, csum->start,
                              (uint8_t)(((int)old_csum & 0xff) + delta));
      }
      break;

    case low_first:
      {
        uint16_t old_csum;
        uint8_t old_csum_low;
        uint8_t old_csum_high;
        uint16_t new_csum;
        uint8_t new_csum_low;
        uint8_t new_csum_high;

        old_csum_low = cmos_read_byte (bank->address_port, bank->data_port, csum->start);
        old_csum_high = cmos_read_byte (bank->address_port, bank->data_port, csum->start + 1);
        old_csum = ((uint16_t)old_csum_high << 8) | old_csum_low;
        new_csum = (uint16_t)(((int)old_csum & 0xffff) + delta);
        new_csum_low = (uint8_t)(new_csum & 0xff);
        new_csum_high = (uint8_t)(new_csum >> 8);
        cmos_fake_write_byte (bank->address_port, bank->data_port, csum->start, new_csum_low);
        cmos_fake_write_byte (bank->address_port, bank->data_port, csum->start + 1, new_csum_high);
      }
      break;

    case high_first:
      {
        uint16_t old_csum;
        uint8_t old_csum_low;
        uint8_t old_csum_high;
        uint16_t new_csum;
        uint8_t new_csum_low;
        uint8_t new_csum_high;

        old_csum_low = cmos_read_byte (bank->address_port, bank->data_port, csum->start + 1);
        old_csum_high = cmos_read_byte (bank->address_port, bank->data_port, csum->start);
        old_csum = ((uint16_t)old_csum_high << 8) | old_csum_low;
        new_csum = (uint16_t)(((int)old_csum & 0xffff) + delta);
        new_csum_low = (uint8_t)(new_csum & 0xff);
        new_csum_high = (uint8_t)(new_csum >> 8);
        cmos_fake_write_byte (bank->address_port, bank->data_port, csum->start + 1, new_csum_low);
        cmos_fake_write_byte (bank->address_port, bank->data_port, csum->start, new_csum_high);
      }
      break;

    default:
      error (2, 0, "cmos_fake_update_checksum: can't happen (internal error, please report)");
    }
}

/* cmos_update_checksums - fake update a list of checksums */
/* csums = list of checksums wrapped in backpatch structures */
/* delta = change in bytes covered by csums, to be added into csums */

static void
cmos_update_checksums (backpatch_t* csums, int delta)
{
  backpatch_t* pbp;

  for (pbp = csums; pbp != NULL; pbp = pbp->next)
    cmos_update_checksum (pbp->u.pchecksum, delta);
}

/* cmos_fake_update_checksums - fake update a list of checksums */
/* csums = list of checksums wrapped in backpatch structures */
/* delta = change in bytes covered by csums, to be added into csums */

static void
cmos_fake_update_checksums (backpatch_t* csums, int delta)
{
  backpatch_t* pbp;

  for (pbp = csums; pbp != NULL; pbp = pbp->next)
    cmos_fake_update_checksum (pbp->u.pchecksum, delta);
}

/* do_cmos_write_op - write a value to a CMOS field described by a layout entry */
/* pe = points to layout entry */
/* value = value to write to the field */

void
do_cmos_write_op (const cmentry_t* pe, uint32_t value)
{
  bank_t* bank;

  bank = pe->bp_bank.u.pbank;
  cmos_set_privs (bank->address_port, bank->data_port, 1);
  cmos_set_checksum_privs (pe->pbp_checksums, 1);
  {
    int first_byte;
    int first_byte_start_bit;
    int first_byte_bits;
    uint8_t first_byte_mask;
    int last_byte;
    int last_byte_bits;
    uint8_t last_byte_mask;

    /* compute bytes covered */
    first_byte = pe->start_bit / 8;
    first_byte_start_bit = pe->start_bit % 8;
    first_byte_bits = safe_min (pe->bit_length, 8 - first_byte_start_bit);
    first_byte_mask = (uint8_t)((1 << first_byte_bits) - 1);

    last_byte = (pe->start_bit + pe->bit_length - 1) / 8;
    if (last_byte > first_byte)
      {
        last_byte_bits = (pe->start_bit + pe->bit_length) % 8;
        last_byte_mask = (uint8_t)((1 << last_byte_bits) - 1);
      }
    {
      int delta;
      
      /* modify first byte */
      delta = cmos_modify_byte (bank->address_port, bank->data_port, first_byte,
                                first_byte_start_bit, first_byte_mask, value);
      cmos_update_checksums (pe->pbp_checksums, delta);
      value >>= first_byte_bits;
      {
        int i;
        
        /* modify bytes second to next last */
        for (i = first_byte + 1; i < last_byte; i++)
          {
            delta = cmos_modify_byte (bank->address_port, bank->data_port, i, 0, 0xff, value);
            cmos_update_checksums (pe->pbp_checksums, delta);
            value >>= 8;
          }
      }
      /* modify last byte if necessary */
      if (last_byte > first_byte)
        {
          delta = cmos_modify_byte (bank->address_port, bank->data_port, last_byte,
                                    0, last_byte_mask, value);
          cmos_update_checksums (pe->pbp_checksums, delta);
        }
    }
  }
  cmos_set_privs (bank->address_port, bank->data_port, 0);
  cmos_set_checksum_privs (pe->pbp_checksums, 0);
}

void
do_cmos_fake_op (const cmentry_t* pe, uint32_t value)
{
  bank_t* bank;

  bank = pe->bp_bank.u.pbank;
  cmos_set_privs (bank->address_port, bank->data_port, 1);
  cmos_set_checksum_privs (pe->pbp_checksums, 1);
  {
    int first_byte;
    int first_byte_start_bit;
    int first_byte_bits;
    uint8_t first_byte_mask;
    int last_byte;
    int last_byte_bits;
    uint8_t last_byte_mask;

    /* compute bytes covered */
    first_byte = pe->start_bit / 8;
    first_byte_start_bit = pe->start_bit % 8;
    first_byte_bits = safe_min (pe->bit_length, 8 - first_byte_start_bit);
    first_byte_mask = (uint8_t)((1 << first_byte_bits) - 1);

    last_byte = (pe->start_bit + pe->bit_length - 1) / 8;
    if (last_byte > first_byte)
      {
        last_byte_bits = (pe->start_bit + pe->bit_length) % 8;
        last_byte_mask = (uint8_t)((1 << last_byte_bits) - 1);
      }
    {
      int delta;
      
      /* modify first byte */
      delta = cmos_fake_modify_byte (bank->address_port, bank->data_port, first_byte,
                                     first_byte_start_bit, first_byte_mask, value);
      cmos_fake_update_checksums (pe->pbp_checksums, delta);
      value >>= first_byte_bits;
      {
        int i;
        
        /* modify bytes second to next last */
        for (i = first_byte + 1; i < last_byte; i++)
          {
            delta = cmos_fake_modify_byte (bank->address_port, bank->data_port, i, 0, 0xff, value);
            cmos_fake_update_checksums (pe->pbp_checksums, delta);
            value >>= 8;
          }
      }
      /* modify last byte if necessary */
      if (last_byte > first_byte)
        {
          delta = cmos_fake_modify_byte (bank->address_port, bank->data_port, last_byte,
                                         0, last_byte_mask, value);
          cmos_fake_update_checksums (pe->pbp_checksums, delta);
        }
    }
  }
  cmos_set_privs (bank->address_port, bank->data_port, 0);
  cmos_set_checksum_privs (pe->pbp_checksums, 0);
}

/* do_cmos_read_op - read a value from a CMOS field described by a layout entry */
/* pe = points to layout entry */
/* return : value read from the field */

uint32_t
do_cmos_read_op (const cmentry_t* pe)
{
  uint32_t value;
  bank_t* bank;

  bank = pe->bp_bank.u.pbank;
  cmos_set_privs (bank->address_port, bank->data_port, 1);
  {
    int first_byte;
    int first_byte_start_bit;
    int first_byte_bits;
    uint8_t first_byte_mask;
    int last_byte;
    int last_byte_bits;
    uint8_t last_byte_mask;

    /* compute bytes covered */
    first_byte = pe->start_bit / 8;
    first_byte_start_bit = pe->start_bit % 8;
    first_byte_bits = safe_min (pe->bit_length, 8 - first_byte_start_bit);
    first_byte_mask = (uint8_t)((1 << first_byte_bits) - 1);

    last_byte = (pe->start_bit + pe->bit_length - 1) / 8;
    if (last_byte > first_byte)
      {
        last_byte_bits = (pe->start_bit + pe->bit_length) % 8;
        last_byte_mask = (uint8_t)((1 << last_byte_bits) - 1);
      }
    /* read first byte */
    value = cmos_extract_byte (bank->address_port, bank->data_port, first_byte,
                               first_byte_start_bit, first_byte_mask);
    {
      int i;
      int shift;
        
      /* read bytes second to next last */
      shift = first_byte_bits;
      for (i = first_byte + 1; i < last_byte; i++)
        {
          value |= (cmos_extract_byte (bank->address_port, bank->data_port, i, 0, 0xff) << shift);
          shift += 8;
        }

      /* read last byte if necessary */
      if (last_byte > first_byte)
        value |= (cmos_extract_byte (bank->address_port, bank->data_port, last_byte,
                                     0, last_byte_mask) << shift);
    }
  }
  cmos_set_privs (bank->address_port, bank->data_port, 0);
  return value;
}

/* cmos_write_region - dump a region of CMOS memory to a file */
/* preg = points to region to dump */
/* fp = file handle to write the dump to */
/* fname = name of open file (for error reporting purposes) */

void
cmos_write_region (const region_t* preg, FILE* fp, const char* fname)
{
  uint8_t buf[1 << 8];
  bank_t* bank;

  bank = preg->bp_bank.u.pbank;
  cmos_set_privs (bank->address_port, bank->data_port, 1);
  {
    int i;
    int last_byte;
    uint8_t* bufp;

    last_byte = preg->start_byte + preg->byte_length - 1;
    bufp = buf;
    for (i = preg->start_byte; i <= last_byte; i++, bufp++)
      *bufp = cmos_read_byte (bank->address_port, bank->data_port, (uint8_t)(i & 0xff));
  }
  cmos_set_privs (bank->address_port, bank->data_port, 0);
  if (fwrite (buf, 1, preg->byte_length, fp) < preg->byte_length)
    error (1, errno, "cmos_write_dump: %s", fname);
}

/* cmos_read_region - read a region of CMOS memory from a file dump */
/* preg = points to region to undump */
/* fp = file handle to read the dump from */
/* fname = name of open file (for error reporting purposes) */

void
cmos_read_region (const region_t* preg, FILE* fp, const char* fname)
{
  uint8_t buf[1 << 8];
  bank_t* bank;

  bank = preg->bp_bank.u.pbank;
  if (fread (buf, 1, preg->byte_length, fp) < preg->byte_length)
    error (1, errno, "cmos_read_dump: %s", fname);
  cmos_set_privs (bank->address_port, bank->data_port, 1);
  {
    int i;
    int last_byte;
    uint8_t* bufp;

    last_byte = preg->start_byte + preg->byte_length - 1;
    bufp = buf;
    for (i = preg->start_byte; i <= last_byte; i++, bufp++)
      cmos_write_byte (bank->address_port, bank->data_port, (uint8_t)(i & 0xff), *bufp);
  }
  cmos_set_privs (bank->address_port, bank->data_port, 0);
}
