
/* This file is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option) */
/* any later version. */

/* This file is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with GNU Emacs; see the file COPYING.  If not, write to */
/* the Free Software Foundation, Inc., 59 Temple Place - Suite 330, */
/* Boston, MA 02111-1307, USA. */

/* Copyright (C) 2004 California Digital Corporation */
/* $Id: biosconfig.c,v 1.22 2004/04/30 22:38:38 summerisle Exp $ */

#include "regexp.h"
#include "valuefile.h"
#include "cmosop.h"
#include "layoutlex.h"
#include "object.h"
#include "smbios.h"
#include "md5.h"
#include "escape.h"
#include "options.h"

#ifdef HAVE_ARGP_H
#include <argp.h>
#else
#include <unistd.h>
#endif

#ifdef HAVE_ERROR_H
#include <error.h>
#endif

#include <regex.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <inttypes.h>

#if (!defined(strndup) && !defined(HAVE_STRNDUP))
extern char* strndup (const char* str, size_t size);
#endif

struct work_params;
typedef struct work_params work_params_t;
typedef void (*work_t)(work_params_t*, layout_t*);

enum work_flags
  {
    work_no_name = 1,
/*     work_parse_stdin = 2, */
/*     work_dry_run = 4, */
    work_force_config = 8,
  };

struct work_params
{
  work_t work;
  enum work_flags flags;
  char* param;
  char* value;
  char* fname;
  char* config;
};

static void hash_bios_info (char** vec, char hash[33]);

static void work_read (work_params_t*, layout_t*);
static void work_enumerate (work_params_t*, layout_t*);
static void work_read_all (work_params_t*, layout_t*);
static void work_write (work_params_t*, layout_t*);
static void work_parse (work_params_t*, layout_t*);
static void work_write_dump (work_params_t*, layout_t*);
static void work_read_dump (work_params_t*, layout_t*);
static void work_hash_info (work_params_t*, layout_t*);

static int
process_option (int key, char* arg, work_params_t* pparams)
{
  switch (key)
    {
      /* this is just a hack for the maintainer
         to easily retrieve SMBIOS identification strings */
    case 'P':
      {
        char* vec[3];
        int i;
        
        copy_bios_info_strings (vec);
        for (i = 0; i < 3; i++)
          {
            if (vec[i] != NULL)
              {
                putc ('"', stdout);
                file_escape_string (stdout, vec[i], "\"");
                putc ('"', stdout);
              }
            putc ('\n', stdout);
          }
        exit (0);
      }

    case 'H':
      if (pparams->work) goto failure;
      pparams->work = work_hash_info;
      pparams->flags |= work_force_config;
      break;

    case 'D':
      printf ("%s\n", PKGCONFDIR);
      exit (0);

    case 'F':
      pparams->flags |= work_force_config;
      break;

    case 'r':
      if (pparams->work) goto failure;
      pparams->work = work_read;
      pparams->param = arg;
      break;

    case 'n':
      pparams->flags |= work_no_name;
      break;

    case 'e':
      if (pparams->work) goto failure;
      pparams->work = work_enumerate;
      pparams->param  = arg;
      break;

    case 't':
      if (pparams->work) goto failure;
      pparams->work = work_read_all;
      pparams->fname = arg;
      break;

    case 'w':
      if (pparams->work) goto failure;
      else
        {
          regmatch_t matches[3];

          if (regexec (&assignment_regex, arg, 3, matches, 0) != 0)
            return -4;
          pparams->param = strndup (arg + matches[1].rm_so, matches[1].rm_eo - matches[1].rm_so);
          pparams->value = strndup (arg + matches[2].rm_so, matches[2].rm_eo - matches[2].rm_so);
          if (!pparams->param || !pparams->value)
            return -3;
          pparams->work = work_write;
        }
      break;

    case 'f':
      if (pparams->work) goto failure;
      pparams->work = work_parse;
      pparams->fname = arg;
      break;

    case 'd':
      if (pparams->work) goto failure;
      pparams->work = work_write_dump;
      pparams->fname = arg;
      break;

    case 'C':
      pparams->config = arg;
      break;

    case 'u':
      if (pparams->work) goto failure;
      pparams->work = work_read_dump;
      pparams->fname = arg;
      break;
    failure:
      return -2;
  
    default:
      return -1;

    }
  return 0;
}

#if (defined(HAVE_ARGP_H) && defined(HAVE_ARGP_PARSE))

error_t
parse_options (int key, char *arg, struct argp_state *state)
{
  work_params_t* pparams;

  pparams = (work_params_t*)(state->input);
  switch (process_option (key, arg, pparams))
    {
    case -1:
      return ARGP_ERR_UNKNOWN;

    case -2:
      argp_error (state, "-%c: only one operation type is permitted\n", key);
      
    case -3:
      argp_failure (state, argp_err_exit_status, errno, "strndup");

    case -4:
      argp_error (state, "`%s' is invalid: --write argument must look like `NAME=VALUE'\n", arg);

    default:
      error (2, 0, "can't happen (internal error, please report)");

    case 0:
      return 0;
    }
}

#else

static void
parse_options_w_getopt (int argc, char** argv, work_params_t* pparams)
{
  int key;

  while ((key = getopt (argc, argv, getopt_options_list)) != EOF)
    {
      switch (process_option (key, optarg, pparams))
        {
        case 0:
          continue;

        case -1:
          error (1, 0,
                 "-%c: unrecognized option, please read the documentation\n"
                 "long options are not avaliable because your system lacks argp\n",
                 optopt);

        case -2:
          error (1, 0, "-%c: only one operation type is permitted\n", key);
      
        case -3:
          error (1, errno, "strndup");

        case -4:
          error (1, 0, "`%s' is invalid: -w argument must look like `NAME=VALUE'\n", optarg);

        default:
          error (2, 0, "can't happen (internal error, please report)");
        }
    }
}

#endif

int
main (int argc, char** argv)
{
  work_params_t the_params;
  layout_t* pl;
  char* vec [3];
  char hash [33];
  bios_info_t* infop;

  regexp_init ();
  
  the_params.work = NULL;
  the_params.flags = 0;
  the_params.param = NULL;
  the_params.value = NULL;
  the_params.fname = NULL;
  the_params.config = NULL;

#ifdef HAVE_ARGP_PARSE
  argp_program_version = PACKAGE_VERSION;
  argp_program_bug_address = PACKAGE_BUGREPORT;
  if (argp_parse (&options_parser, argc, argv, 0, 0, &the_params) != 0)
    error (argp_err_exit_status, 0, "argp_parse");
#else
  parse_options_w_getopt (argc, argv, &the_params);
#endif

  vec [0] = vec [1] = vec [2] = NULL;
  if ((the_params.flags & work_force_config) == 0 || !the_params.config)
    {
      copy_bios_info_strings (vec);
      hash_bios_info (vec, hash);
    }
  if (!the_params.config)
    {
      int len;

      len = strlen (PKGCONFDIR) + 40;
      the_params.config = malloc (len);
      if (!the_params.config) error (1, errno, "malloc");
      snprintf (the_params.config, len, "%s/%s.tok", PKGCONFDIR, hash);
    }
  pl = parse_file (the_params.config);
  for (infop = pl->compat; infop != NULL; infop = infop->next)
    {
      if (infop->vendor == NULL && vec [0] != NULL) continue;
      if (infop->vendor != NULL && vec [0] == NULL) continue;
      if (vec [0] != NULL && strcmp (infop->vendor, vec [0]) != 0) continue;
      if (infop->version == NULL && vec [1] != NULL) continue;
      if (infop->version != NULL && vec [1] == NULL) continue;
      if (vec [1] != NULL && strcmp (infop->version, vec [1]) != 0) continue;
      if (infop->date == NULL && vec [2] != NULL) continue;
      if (infop->date != NULL && vec [2] == NULL) continue;
      if (vec [2] != NULL && strcmp (infop->date, vec[2]) != 0) continue;
      break;
    }
  if (infop == NULL && (the_params.flags & work_force_config) == 0)
    error (3, 0, "Probed BIOS version not covered by %s", the_params.config);
  analyze_layout (pl);

  if (!the_params.work) return 0;
  (*the_params.work)(&the_params, pl);
  return 0;
}

/* work_read - handle --read option */
/* pparams = pointer to structure collecting command line arguments */
/* pl = pointer to structure representing parsed configuration file */

static void
work_read (work_params_t* pparams, layout_t* pl)
{
  uint32_t value;
  cmentry_t* pe;

  pe = find_entry (pparams->param, pl);
  value = do_cmos_read_op (pe);
  if ((pparams->flags & work_no_name) == 0)
    printf ("%s = ", pparams->param);
  print_value (stdout, pe, value, pl);
  printf ("\n");  
}

/* work_enumerate - handle --enumerate option */
/* pparams = pointer to structure collecting command line arguments */
/* pl = pointer to structure representing parsed configuration file */

static void
work_enumerate (work_params_t* pparams, layout_t* pl)
{
  cmentry_t* pe;
  enumeration_t* penum;
  enum_value_t* pev;

  pe = find_entry (pparams->param, pl);
  if (pe->type != enumtype)
    error (1, 0, "entry %s is not an enumeration", pe->name->name);
  penum = pe->bp_enum.u.penum;
  for (pev = penum->values; pev != NULL; pev = pev->next)
    printf ("%s\n", pev->name->name);
}

/* work_read_all - handle --to-file option */
/* unused = pointer to structure collecting command line arguments */
/* pl = pointer to structure representing parsed configuration file */

static void
work_read_all (work_params_t* pparams, layout_t* pl)
{
  cmentry_t* pe;
  FILE* fp;

  if (strcmp (pparams->fname, "-") == 0)
    for (pe = pl->entries; pe != NULL; pe = pe->next)
      {
        uint32_t value;
        
        value = do_cmos_read_op (pe);
        printf ("%s = ", pe->name->name);
        print_value (stdout, pe, value, pl);
        printf ("\n");
      }
  else
    {
      fp = fopen (pparams->fname, "w");
      if (fp == NULL) error (1, errno, "%s", pparams->fname);
      for (pe = pl->entries; pe != NULL; pe = pe->next)
        {
          uint32_t value;
        
          value = do_cmos_read_op (pe);
          fprintf (fp, "%s = ", pe->name->name);
          print_value (fp, pe, value, pl);
          fprintf (fp, "\n");
        }
      fclose (fp);
    }
}

/* work_write - handle --write option */
/* pparams = pointer to structure collecting command line arguments */
/* pl = pointer to structure representing parsed configuration file */

static void
work_write (work_params_t* pparams, layout_t* pl)
{
  cmentry_t* pe;
  uint32_t value;

  pe = find_entry (pparams->param, pl);
  value = lookup_value (pe, pparams->value, pl);
  do_cmos_write_op (pe, value);
}

/* work_parse - handle --from-file option */
/* pparams = pointer to structure collecting command line arguments */
/* pl = pointer to structure representing parsed configuration file */

static void
work_parse (work_params_t* pparams, layout_t* lp)
{
  value_pair_t* pvp;
  value_pair_t* pvp1;

  pvp = read_value_file (pparams->fname);
  for (pvp1 = pvp; pvp1 != NULL; pvp1 = pvp1->next)
    {
      cmentry_t* pe;
      uint32_t value;

      pe = find_entry (pvp1->name, lp);
      value = lookup_value (pe, pvp1->value, lp);
      do_cmos_write_op (pe, value);
    }
}

/* work_write_dump - handle --dump option */
/* pparams = pointer to structure collecting command line arguments */
/* pl = pointer to structure representing parsed configuration file */

static void
work_write_dump (work_params_t* pparams, layout_t* lp)
{
  FILE* fp;

  if (!(fp = fopen (pparams->fname, "w")))
    error (1, errno, "work_write_dump: %s", pparams->fname);
  {
    region_t* preg;

    for (preg = lp->regions; preg != NULL; preg = preg->next)
      cmos_write_region (preg, fp, pparams->fname);
  }
  fclose (fp);
}

/* work_read_dump - handle --undump option */
/* pparams = pointer to structure collecting command line arguments */
/* pl = pointer to structure representing parsed configuration file */

static void
work_read_dump (work_params_t* pparams, layout_t* lp)
{
  FILE* fp;

  if (!(fp = fopen (pparams->fname, "r")))
    error (1, errno, "work_read_dump: %s", pparams->fname);
  {
    region_t* preg;

    for (preg = lp->regions; preg != NULL; preg = preg->next) /* FIXME */
      cmos_read_region (preg, fp, pparams->fname);
  }
  fclose (fp);
}

/* hash_bios_info - create a hex digest of BIOS identification strings */
/* vec = vector of identification strings: vendor, version, date */
/* hash = string to place digest in */

static void
hash_bios_info (char** vec, char hash[33])
{
  md5_state_t state;
  int i;
  md5_byte_t digest [16];
  static const char hex [] = "0123456789abcdef";

  md5_init (&state);
  for (i = 0; i < 3; i++)
    {
      if (vec [i]) md5_append (&state, vec [i], strlen (vec [i]));
      md5_append (&state, "\n", 1);
    }
  md5_finish (&state, digest);
  for (i = 0; i < 16; i++)
    {
      hash [2 * i] = hex [(digest [i] >> 4) & 0xf];
      hash [(2 * i) + 1] = hex [digest[i] & 0xf];
    }
  hash [32] = '\0';
}

/* work_hash_info - handle --hash option */
/* pparams = pointer to structure collecting command line arguments */
/* pl = pointer to structure representing parsed configuration file */

static void
work_hash_info (work_params_t* pparams, layout_t* lp)
{
  bios_info_t* infop;

  for (infop = lp->compat; infop != NULL; infop = infop->next)
    {
      char hash [33];
      char* vec[3];

      vec [0] = infop->vendor;
      vec [1] = infop->version;
      vec [2] = infop->date;
      hash_bios_info (vec, hash);
      printf ("%s\n", hash);
    }
}
