# GNU Shepherd --- Test one-shot services.
# Copyright © 2019, 2023-2025 Ludovic Courtès <ludo@gnu.org>
#
# This file is part of the GNU Shepherd.
#
# The GNU Shepherd is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or (at
# your option) any later version.
#
# The GNU Shepherd is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with the GNU Shepherd.  If not, see <http://www.gnu.org/licenses/>.

shepherd --version
herd --version

socket="t-socket-$$"
conf="t-conf-$$"
confdir="t-confdir-$$"
log="t-log-$$"
stamp="t-stamp-$$"
pid="t-pid-$$"

herd="herd -s $socket"

trap "cat $log || true; rm -f $socket $conf $stamp $log;
      test -f $pid && kill \`cat $pid\` || true; rm -f $pid" EXIT

cat > "$conf"<<EOF
(use-modules (srfi srfi-26))
(register-services
 (list (service
	 '(always-fail)
	 #:start (const #f)
	 #:one-shot? #t)
       (service
	 '(test)
	 #:start (lambda _
		   (call-with-output-file "$stamp"
		     (cut display "foo" <>))
		   #t)
	 #:one-shot? #t)
       (service
	 '(test-2)
	 #:requirement '(test)
	 #:start (lambda _
		   (call-with-output-file "$stamp-2"
		     (cut display "bar" <>))
		   #t)
	 #:stop  (lambda _
		   (delete-file "$stamp-2")
		   #f))

       ;; Several services depending on the same one-shot service.
       (service
	 '(one-shotty)
	 #:start (const #t)
	 #:one-shot? #t)
       (service
	 '(a)
	 #:requirement '(one-shotty)
	 #:start (const #t))
       (service
	 '(b)
	 #:requirement '(a one-shotty)
	 #:start (const #t))
       (service
	 '(c)
	 #:requirement '(a b one-shotty)
	 #:start (const #t))))

(start-in-the-background '(a b c))
EOF

rm -f "$pid"
shepherd -I -s "$socket" -c "$conf" -l "$log" --pid="$pid" &

# Wait till it's ready.
while ! test -f "$pid" ; do sleep 0.3 ; done

shepherd_pid="`cat $pid`"

kill -0 $shepherd_pid
test -S "$socket"

# Make sure we notice startup failures of one-shot services.
if $herd start always-fail; then false; else true; fi

for i in 1 2 3
do
    rm -f "$stamp"
    $herd start test
    test -f "$stamp"
    $herd status test | grep stopped.*one-shot
    grep "test.*started" "$log"
    $herd stop test		# no-op since it's already stopped
done

$herd status | grep -i '^one-shot'
$herd status

rm -f "$stamp" "$stamp-2"
$herd start test-2
test -f "$stamp"
test -f "$stamp-2"
$herd status test | grep stopped.*one-shot
$herd status test-2 | grep running
$herd stop test-2
if test -f "$stamp-2"; then false; else true; fi

# When starting A, B, and C via 'start-in-the-background', ONE-SHOTTY should
# have been started once only.
test $(grep "Starting service one-shotty" "$log" | wc -l) -eq 1
$herd stop a

# In the course of starting C, ONE-SHOTTY should be started only once.
$herd start c
test $(grep "Starting service one-shotty" "$log" | wc -l) -eq 2

# But we can still start it a second time, indirectly...
$herd stop a
$herd start c
test $(grep "Starting service one-shotty" "$log" | wc -l) -eq 3

# ... and a third time, directly.
$herd start one-shotty
test $(grep "Starting service one-shotty" "$log" | wc -l) -eq 4

# Now test dependencies among one-shot services: they should be started in the
# right order.  See <https://issues.guix.gnu.org/74284>.

cat > "$conf" <<EOF
(use-modules (srfi srfi-26) (ice-9 textual-ports))

;; Each service below depends on the previous one and must thus
;; be started after it.  To verify that ordering is respected,
;; arrange so that each 'start' method takes one second.

(register-services
 (list (service
	 '(first)
         #:one-shot? #t
	 #:start (lambda ()
                   (sleep 1)
                   (call-with-output-file "$stamp"
                     (cut display "first" <>))
                   #t))
       (service
	 '(second)
         #:one-shot? #t
	 #:requirement '(first)
	 #:start (lambda ()
                   (unless (string=? "first"
                                     (call-with-input-file "$stamp"
                                       get-string-all))
                     (error "first not started!"))
                   (delete-file "$stamp")
                   (sleep 1)
                   (call-with-output-file "$stamp"
                     (cut display "second" <>))
                   #t))
       (service
	 '(third)
         #:one-shot? #t
	 #:requirement '(second)
	 #:start (lambda ()
                   (unless (string=? "second"
                                     (call-with-input-file "$stamp"
                                       get-string-all))
                     (error "second not started!"))
                   (delete-file "$stamp")
                   (call-with-output-file "$stamp"
                     (cut display "third" <>))
                   #t))
       (service
         '(fourth)
         #:one-shot? #t
         #:requirement '(always-fail)
         #:start (const #t))))

;; Cause services to be started potentially in parallel.  In this case,
;; they should be started one after another due to dependencies.
(start-in-the-background '(first second third))
EOF

rm -f "$stamp"
$herd load root "$conf"

until grep "Service third started" "$log"; do sleep 0.5; done
test "$(cat "$stamp")" = "third"

$herd start fourth && false
$herd start fourth && false

# Check the behavior of two clients competing to start the same one-shot
# service.  Both should succeed.

cat > "$conf" <<EOF
(register-services
  (list (service
          '(fifth)
          #:one-shot? #t
          #:start (lambda ()
                    (let loop ()
                      (unless (file-exists? "$stamp")
                        (sleep 0.5)
                        (loop)))
                    #t))))
EOF

$herd load root "$conf"

rm -f "$stamp"

$herd start fifth &
herd_start_pid1=$!
$herd start fifth &
herd_start_pid2=$!
until $herd status fifth | grep starting; do sleep 0.5; done
touch "$stamp"			# trigger starting->running transition

# Both 'herd start' processes should have succeeded.
wait $herd_start_pid1
wait $herd_start_pid2

$herd stop root
